<?php

namespace App\Console\Commands;

use App\Models\SystemSetting;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CheckRenewalPeriod extends Command
{
    protected $signature = 'enrollment:check-period';
    protected $description = 'Abre ou fecha automaticamente o período de renovação de matrícula com base nas datas configuradas';

    public function handle(): int
    {
        $openDate  = SystemSetting::get('renewal_auto_open_date',  null, 'enrollment');  // ex: "01/03"
        $closeDate = SystemSetting::get('renewal_auto_close_date', null, 'enrollment');  // ex: "31/03"

        if (!$openDate && !$closeDate) {
            $this->info('Nenhuma data automática configurada. Nada a fazer.');
            return self::SUCCESS;
        }

        $today = Carbon::now('Africa/Maputo');
        $year  = $today->year;

        $shouldBeOpen = false;

        if ($openDate && $closeDate) {
            // Ambas configuradas: aberto entre as duas datas (inclusive)
            $open  = Carbon::createFromFormat('d/m/Y', $openDate  . '/' . $year, 'Africa/Maputo')->startOfDay();
            $close = Carbon::createFromFormat('d/m/Y', $closeDate . '/' . $year, 'Africa/Maputo')->endOfDay();
            $shouldBeOpen = $today->between($open, $close);
        } elseif ($openDate) {
            // Só data de abertura: abre nesse dia e fica aberto
            $open = Carbon::createFromFormat('d/m/Y', $openDate . '/' . $year, 'Africa/Maputo')->startOfDay();
            $shouldBeOpen = $today->gte($open);
        } elseif ($closeDate) {
            // Só data de fecho: fecha nesse dia
            $close = Carbon::createFromFormat('d/m/Y', $closeDate . '/' . $year, 'Africa/Maputo')->endOfDay();
            $shouldBeOpen = $today->lte($close);
        }

        $currentlyOpen = (bool) SystemSetting::get('renewal_period_open', false, 'enrollment');

        if ($shouldBeOpen === $currentlyOpen) {
            $this->info('Estado do período já está correcto (' . ($currentlyOpen ? 'aberto' : 'fechado') . '). Sem alterações.');
            return self::SUCCESS;
        }

        SystemSetting::set(
            'renewal_period_open',
            $shouldBeOpen ? '1' : '0',
            'boolean',
            'enrollment',
            'Período de renovação aberto'
        );
        SystemSetting::clearGroupCache('enrollment');

        $msg = $shouldBeOpen
            ? 'Período de renovação ABERTO automaticamente.'
            : 'Período de renovação FECHADO automaticamente.';

        $this->info($msg);
        Log::channel('daily')->info('[enrollment:check-period] ' . $msg);

        return self::SUCCESS;
    }
}
