<?php

namespace App\Console\Commands;

use App\Services\SmsService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class SendGuardianCredentials extends Command
{
    protected $signature = 'guardians:send-credentials
                            {--only-new : Enviar apenas para guardians que nunca fizeram login}
                            {--dry-run : Simular sem enviar SMS}
                            {--phone= : Enviar apenas para um telefone específico}
                            {--generate : Gerar guardians a partir de pre_enrollments antes de enviar}';

    protected $description = 'Gera senhas e envia credenciais de acesso ao portal por SMS aos encarregados';

    private int $created = 0;
    private int $smsSent = 0;
    private int $smsFailed = 0;
    private int $skipped = 0;

    public function handle(): int
    {
        $this->info('=== Credenciais do Portal de Encarregados ===');
        $this->newLine();

        // Step 1: Gerar guardians se pedido
        if ($this->option('generate')) {
            $this->generateGuardiansFromPreEnrollments();
            $this->newLine();
        }

        // Step 2: Gerar senhas novas e enviar por SMS
        $this->sendCredentials();

        // Resumo
        $this->newLine();
        $this->info('=== Resumo ===');
        if ($this->option('generate')) {
            $this->line("  Guardians criados: {$this->created}");
        }
        $this->line("  SMS enviados: {$this->smsSent}");
        $this->line("  SMS falharam: {$this->smsFailed}");
        $this->line("  Ignorados: {$this->skipped}");

        return Command::SUCCESS;
    }

    private function generateGuardiansFromPreEnrollments(): void
    {
        $this->info('Gerando guardians a partir de pre-inscricoes...');

        $preEnrollments = DB::table('pre_enrollments')
            ->whereNotNull('resulting_user_id')
            ->whereIn('status', ['matriculada'])
            ->get();

        $bar = $this->output->createProgressBar($preEnrollments->count());

        foreach ($preEnrollments as $pe) {
            // Pai
            if (!empty($pe->father_phone) && !empty($pe->father_name)) {
                $this->createGuardian($pe->father_name, $pe->father_phone, 'pai', $pe->resulting_user_id);
            }

            // Mae (se telefone diferente do pai)
            if (!empty($pe->mother_phone) && !empty($pe->mother_name)) {
                if ($pe->mother_phone !== ($pe->father_phone ?? '')) {
                    $this->createGuardian($pe->mother_name, $pe->mother_phone, 'mae', $pe->resulting_user_id);
                }
            }

            // Encarregado (se diferente de pai e mae)
            if (!empty($pe->guardian_phone) && !empty($pe->guardian_name)) {
                if ($pe->guardian_phone !== ($pe->father_phone ?? '') && $pe->guardian_phone !== ($pe->mother_phone ?? '')) {
                    $relationship = $pe->guardian_relationship ?? 'encarregado';
                    $this->createGuardian($pe->guardian_name, $pe->guardian_phone, $relationship, $pe->resulting_user_id);
                }
            }

            $bar->advance();
        }

        $bar->finish();
        $this->newLine();
        $this->info("Guardians criados: {$this->created}");
    }

    private function createGuardian(string $name, string $phone, string $relationship, string $studentId): void
    {
        $phone = preg_replace('/\D/', '', $phone);
        if (empty($phone)) return;

        if (strlen($phone) === 9 && !str_starts_with($phone, '258')) {
            $phone = '258' . $phone;
        }

        $existing = DB::table('guardians')->where('phone', $phone)->first();

        if (!$existing) {
            $guardianId = (string) Str::uuid();
            // Gerar senha aleatoria igual ao sistema de estudantes: rand(100000, 500000)
            $plainPassword = (string) rand(100000, 500000);

            DB::table('guardians')->insert([
                'id'           => $guardianId,
                'name'         => $name,
                'phone'        => $phone,
                'relationship' => $relationship,
                'password'     => Hash::make($plainPassword),
                'is_active'    => true,
                'created_at'   => now(),
                'updated_at'   => now(),
            ]);

            // Guardar senha temporariamente para envio SMS
            DB::table('guardians')->where('id', $guardianId)->update([
                'remember_token' => $plainPassword,
            ]);

            $this->linkGuardianToStudent($guardianId, $studentId, $relationship, true);
            $this->created++;
        } else {
            $this->linkGuardianToStudent($existing->id, $studentId, $relationship, false);
        }
    }

    private function linkGuardianToStudent(string $guardianId, string $studentId, string $relationship, bool $isPrimary): void
    {
        $exists = DB::table('guardian_student')
            ->where('guardian_id', $guardianId)
            ->where('student_id', $studentId)
            ->exists();

        if (!$exists) {
            DB::table('guardian_student')->insert([
                'id'           => (string) Str::uuid(),
                'guardian_id'  => $guardianId,
                'student_id'   => $studentId,
                'relationship' => $relationship,
                'is_primary'   => $isPrimary,
                'created_at'   => now(),
                'updated_at'   => now(),
            ]);
        }
    }

    private function sendCredentials(): void
    {
        $this->info('Gerando senhas e enviando credenciais por SMS...');

        $query = DB::table('guardians')
            ->where('is_active', true)
            ->whereNull('deleted_at');

        // Filtro: apenas quem nunca fez login
        if ($this->option('only-new')) {
            $query->whereNull('last_login_at');
        }

        // Filtro: telefone especifico
        if ($phone = $this->option('phone')) {
            $phone = preg_replace('/\D/', '', $phone);
            if (strlen($phone) === 9) $phone = '258' . $phone;
            $query->where('phone', $phone);
        }

        $guardians = $query->get();

        if ($guardians->isEmpty()) {
            $this->warn('Nenhum encarregado encontrado.');
            return;
        }

        $this->info("Total: {$guardians->count()} encarregados");

        if ($this->option('dry-run')) {
            $this->warn('MODO DRY-RUN: SMS nao serao enviados.');
        }

        $bar = $this->output->createProgressBar($guardians->count());
        $smsService = $this->option('dry-run') ? null : new SmsService();

        foreach ($guardians as $guardian) {
            $bar->advance();

            // Buscar nomes dos estudantes
            $students = DB::table('guardian_student')
                ->join('users', 'users.id', '=', 'guardian_student.student_id')
                ->where('guardian_student.guardian_id', $guardian->id)
                ->pluck('users.name')
                ->toArray();

            if (empty($students)) {
                $this->skipped++;
                continue;
            }

            $studentNames = implode(', ', array_slice($students, 0, 3));
            if (count($students) > 3) {
                $studentNames .= ' e outros';
            }

            // Gerar nova senha aleatoria (mesmo sistema dos estudantes)
            $plainPassword = (string) rand(100000, 500000);

            // Actualizar senha no banco
            DB::table('guardians')->where('id', $guardian->id)->update([
                'password'   => Hash::make($plainPassword),
                'updated_at' => now(),
            ]);

            // Montar mensagem SMS (mesmo estilo do sistema de estudantes)
            $message = "COPMOZ - Portal do Encarregado\n"
                . "Caro(a) {$guardian->name}, foi registado no portal de encarregados COPMOZ para acompanhar: {$studentNames}.\n"
                . "Telefone: {$guardian->phone}\n"
                . "Senha: {$plainPassword}\n"
                . "Link: https://siga.copmoz.ac.mz/encarregado/login";

            if ($this->option('dry-run')) {
                $this->newLine();
                $this->line("  [{$guardian->phone}] {$guardian->name} -> Senha: {$plainPassword} -> Educandos: {$studentNames}");
                $this->smsSent++;
                continue;
            }

            try {
                $result = $smsService->send($guardian->phone, $message);

                if ($result['success']) {
                    $this->smsSent++;
                    Log::info("Credenciais guardian enviadas para {$guardian->phone} ({$guardian->name})");
                } else {
                    $this->smsFailed++;
                    Log::warning("SMS falhou para guardian {$guardian->phone}: " . ($result['error'] ?? 'Erro'));
                }
            } catch (\Exception $e) {
                $this->smsFailed++;
                Log::error("Erro SMS guardian {$guardian->phone}: " . $e->getMessage());
            }

            // Pausa para nao sobrecarregar a API de SMS
            usleep(200000); // 200ms
        }

        $bar->finish();
        $this->newLine();
    }
}
