<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Helpers\MonthTranslator;
use App\Models\FeeStructure;
use App\Models\FeeDiscount;
use Illuminate\Support\Facades\DB;

class TranslateMonthsToPortuguese extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'months:translate {--dry-run : Run without making changes}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Traduz meses de inglês para português em todas as tabelas';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $dryRun = $this->option('dry-run');

        if ($dryRun) {
            $this->info('🔍 Modo DRY RUN - Nenhuma alteração será feita');
        }

        $this->info('Iniciando tradução de meses...');
        $this->newLine();

        $totalUpdated = 0;

        // Traduzir FeeStructures
        $totalUpdated += $this->translateFeeStructures($dryRun);

        $this->newLine();
        if ($dryRun) {
            $this->info("✅ {$totalUpdated} registros SERIAM atualizados (dry-run)");
            $this->warn('Execute sem --dry-run para aplicar as mudanças');
        } else {
            $this->info("✅ {$totalUpdated} registros atualizados com sucesso!");
        }

        return 0;
    }

    private function translateFeeStructures($dryRun)
    {
        $this->info('📋 Processando FeeStructures...');

        $structures = FeeStructure::whereNotNull('months')->get();
        $updated = 0;

        foreach ($structures as $structure) {
            $originalMonths = $structure->months;
            $translatedMonths = MonthTranslator::translateMultiple($originalMonths);

            if ($originalMonths !== $translatedMonths) {
                $this->line("  ID {$structure->id}: {$originalMonths} → {$translatedMonths}");

                if (!$dryRun) {
                    $structure->update(['months' => $translatedMonths]);
                }
                $updated++;
            }
        }

        $this->info("  ✓ {$updated} estruturas de taxas processadas");
        return $updated;
    }

    private function translateFeeDiscounts($dryRun)
    {
        $this->info('💰 Processando FeeDiscounts...');

        $discounts = FeeDiscount::whereNotNull('applicable_months')->get();
        $updated = 0;

        foreach ($discounts as $discount) {
            $originalMonths = $discount->applicable_months;
            $translatedMonths = MonthTranslator::translateMultiple($originalMonths);

            if ($originalMonths !== $translatedMonths) {
                $this->line("  ID {$discount->id}: {$originalMonths} → {$translatedMonths}");

                if (!$dryRun) {
                    $discount->update(['applicable_months' => $translatedMonths]);
                }
                $updated++;
            }
        }

        $this->info("  ✓ {$updated} descontos processados");
        return $updated;
    }

    private function translatePaymentReferences($dryRun)
    {
        $this->info('📝 Processando Payment References...');

        $references = DB::table('payment_references')
            ->whereNotNull('month')
            ->get();
        $updated = 0;

        foreach ($references as $ref) {
            $originalMonth = $ref->month;
            $translatedMonth = MonthTranslator::toPortuguese($originalMonth);

            if ($originalMonth !== $translatedMonth) {
                $this->line("  ID {$ref->id}: {$originalMonth} → {$translatedMonth}");

                if (!$dryRun) {
                    DB::table('payment_references')
                        ->where('id', $ref->id)
                        ->update(['month' => $translatedMonth]);
                }
                $updated++;
            }
        }

        $this->info("  ✓ {$updated} referências de pagamento processadas");
        return $updated;
    }
}
