<?php

namespace App\Console;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

class Kernel extends ConsoleKernel
{
    /**
     * Define the application's command schedule.
     */
    
    protected function schedule(Schedule $schedule): void
    {
        // Mensagens de aniversário
        $schedule->command('send:birthday-messages')
            ->dailyAt('08:00')
            ->timezone('Africa/Maputo')
            ->withoutOverlapping()
            ->appendOutputTo(storage_path('logs/scheduler.log'));

        // Lembretes de pagamento (horário configurável via admin)
        $schedule->command('payment:send-reminders')
            ->hourly()
            ->timezone('Africa/Maputo')
            ->withoutOverlapping()
            ->appendOutputTo(storage_path('logs/payment-reminders.log'))
            ->emailOutputOnFailure(config('mail.admin_email', 'admin@copmoz.co.mz'));

        // Processamento automático de arquivos BMEPS/MEPS
        $schedule->command('bmeps:process')
            ->everyFiveMinutes()
            ->timezone('Africa/Maputo')
            ->withoutOverlapping()
            ->appendOutputTo(storage_path('logs/bmeps-auto.log'))
            ->emailOutputOnFailure(config('mail.admin_email', 'admin@copmoz.co.mz'));

        // Limpeza automática de ficheiros BMEPS processados (>7 dias)
        $schedule->command('bci:clean --days=7')
            ->daily()
            ->timezone('Africa/Maputo')
            ->appendOutputTo(storage_path('logs/bci-clean.log'));

        // Monitorização SFTP: alerta SMS se ficheiros do banco pararem de chegar
        $schedule->command('bmeps:monitor')
            ->everyFourHours()
            ->timezone('Africa/Maputo')
            ->between('06:00', '22:00')
            ->withoutOverlapping()
            ->appendOutputTo(storage_path('logs/bmeps-monitor.log'));

        // =========================================
        // GERAÇÃO AUTOMÁTICA DE REFERÊNCIAS
        // =========================================

        // Gerar referências no dia configurado em payment_start_day (default: dia 5)
        // Executa às 08:00 todos os dias, mas só gera se for o dia correcto
        $schedule->command('references:generate')
            ->dailyAt('08:00')
            ->timezone('Africa/Maputo')
            ->withoutOverlapping()
            ->appendOutputTo(storage_path('logs/references-generate.log'))
            ->emailOutputOnFailure(config('mail.admin_email', 'admin@copmoz.co.mz'));

        // Marcar referências como expiradas às 23:59
        // Executa diariamente e marca as que já passaram da data de vencimento
        $schedule->command('references:expire --no-interaction')
            ->dailyAt('23:59')
            ->timezone('Africa/Maputo')
            ->withoutOverlapping()
            ->appendOutputTo(storage_path('logs/references-expire.log'))
            ->emailOutputOnFailure(config('mail.admin_email', 'admin@copmoz.co.mz'));

        // =========================================
        // APLICAÇÃO DE MULTAS
        // =========================================

        // Aplicar multas às referências vencidas às 06:00
        // Expira referência original e gera nova com multa
        $schedule->command('references:apply-penalties')
            ->dailyAt('06:00')
            ->timezone('Africa/Maputo')
            ->withoutOverlapping()
            ->appendOutputTo(storage_path('logs/references-penalties.log'))
            ->emailOutputOnFailure(config('mail.admin_email', 'admin@copmoz.co.mz'));

        // =========================================
        // PERÍODO DE RENOVAÇÃO DE MATRÍCULA
        // =========================================

        // Verifica diariamente às 00:01 se deve abrir ou fechar o período
        // conforme as datas configuradas em SystemSettings
        $schedule->command('enrollment:check-period')
            ->dailyAt('00:01')
            ->timezone('Africa/Maputo')
            ->withoutOverlapping()
            ->appendOutputTo(storage_path('logs/enrollment-period.log'));
    }

    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
        $this->load(__DIR__.'/Commands');
        require base_path('routes/console.php');
    }
}

