<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithTitle;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;

class ReferencesReportExport implements FromCollection, WithHeadings, WithStyles, WithColumnWidths, WithTitle
{
    protected $references;
    protected $stats;

    public function __construct($references, $stats)
    {
        $this->references = $references;
        $this->stats = $stats;
    }

    public function collection()
    {
        return $this->references->map(function ($ref, $index) {
            return [
                $index + 1,
                $ref->student->name ?? 'N/A',
                $ref->student->student_id ?? 'N/A',
                $ref->classe . 'ª',
                $ref->turma,
                $ref->entity_code,
                $ref->reference_number,
                number_format($ref->amount, 2),
                ucfirst($ref->fee_month),
                $ref->fee_year,
                $ref->academic_year,
                $ref->valid_from->format('d/m/Y'),
                $ref->expires_at->format('d/m/Y'),
                $this->getStatusLabel($ref->status),
                $ref->paid_at ? $ref->paid_at->format('d/m/Y H:i') : '-',
                $ref->created_at->format('d/m/Y H:i'),
            ];
        });
    }

    public function headings(): array
    {
        return [
            '#',
            'Nome do Estudante',
            'Código',
            'Classe',
            'Turma',
            'Entidade',
            'Referência',
            'Valor (MT)',
            'Mês',
            'Ano',
            'Ano Académico',
            'Válido De',
            'Válido Até',
            'Status',
            'Data Pagamento',
            'Data Criação',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        // Adicionar estatísticas no topo
        $sheet->insertNewRowBefore(1, 4);

        $sheet->setCellValue('A1', 'RELATÓRIO DE REFERÊNCIAS DE PAGAMENTO');
        $sheet->setCellValue('A2', 'Data do Relatório: ' . now()->format('d/m/Y H:i'));
        $sheet->setCellValue('A3', '');

        $sheet->setCellValue('A4', 'Total Referências:');
        $sheet->setCellValue('B4', $this->stats['total_references'] ?? 0);
        $sheet->setCellValue('C4', 'Valor Total:');
        $sheet->setCellValue('D4', number_format($this->stats['total_amount'] ?? 0, 2) . ' MT');

        $sheet->setCellValue('F4', 'Pendentes:');
        $sheet->setCellValue('G4', $this->stats['pending'] ?? 0);
        $sheet->setCellValue('H4', 'Pagos:');
        $sheet->setCellValue('I4', $this->stats['paid'] ?? 0);

        // Estilo do título
        $sheet->mergeCells('A1:P1');
        $sheet->getStyle('A1')->applyFromArray([
            'font' => ['bold' => true, 'size' => 16, 'color' => ['rgb' => 'FFFFFF']],
            'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '0066CC']],
            'alignment' => ['horizontal' => 'center', 'vertical' => 'center'],
        ]);

        // Estilo dos cabeçalhos (linha 5 após inserir 4 linhas)
        $sheet->getStyle('A5:P5')->applyFromArray([
            'font' => ['bold' => true, 'size' => 11, 'color' => ['rgb' => 'FFFFFF']],
            'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '4472C4']],
            'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN]],
        ]);

        // Bordas nas células de dados
        $lastRow = $this->references->count() + 5;
        $sheet->getStyle('A5:P' . $lastRow)->applyFromArray([
            'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => 'CCCCCC']]],
        ]);

        return [];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 6,   // #
            'B' => 30,  // Nome
            'C' => 15,  // Código
            'D' => 8,   // Classe
            'E' => 15,  // Turma
            'F' => 12,  // Entidade
            'G' => 15,  // Referência
            'H' => 12,  // Valor
            'I' => 12,  // Mês
            'J' => 8,   // Ano
            'K' => 15,  // Ano Académico
            'L' => 12,  // Válido De
            'M' => 12,  // Válido Até
            'N' => 12,  // Status
            'O' => 18,  // Data Pagamento
            'P' => 18,  // Data Criação
        ];
    }

    public function title(): string
    {
        return 'Referências';
    }

    private function getStatusLabel($status)
    {
        $labels = [
            'pending' => 'Pendente',
            'paid' => 'Pago',
            'expired' => 'Expirado',
            'cancelled' => 'Cancelado',
        ];

        return $labels[$status] ?? ucfirst($status);
    }
}
