<?php
namespace App\Http\Livewire\Admin\Fee;

use App\Models\PaymentReference;
use App\Models\User;
use App\Models\Fee_assign;
use App\Models\FailedReconciliation;
use App\Services\PaymentReconciliationService;
use App\Services\FeeCalculationService;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Collection;

class PaymentReconciliationDashboard extends Component
{
    use WithFileUploads, WithPagination;

    protected $paginationTheme = 'bootstrap';

    // Upload de ficheiros
    public $uploadedFile;
    public $uploadProgress = 0;

    // Filtros
    public $filterStatus = 'all';
    public $filterDateFrom;
    public $filterDateTo;
    public $searchTerm = '';

    // Reconciliação manual
    public $showManualModal = false;
    public $manualReference;
    public $manualAmount;
    public $manualDate;
    public $manualTransactionId;
    public $manualTerminalId;
    public $manualNote;

    // Estatísticas em tempo real
    public $stats = [];

    // Caches para evitar múltiplas consultas
    private $cachedIncomingFiles = null;
    private $cachedUnmatchedPayments = null;
    private $cachedRecentActivity = null;
    private $cachedFailedReconciliations = null;

    // Tab ativo para navegação
    public $activeTab = 'overview';

    protected $reconciliationService;
    protected $feeCalculationService;

    protected $rules = [
        'uploadedFile' => 'required|file|mimes:txt,TXT|max:10240',
        'manualReference' => 'required|string|min:5|max:20',
        'manualAmount' => 'required|numeric|min:0.01',
        'manualDate' => 'required|date',
        'manualTransactionId' => 'nullable|string|max:50',
        'manualTerminalId' => 'nullable|string|max:20',
        'manualNote' => 'nullable|string|max:500',
    ];

    protected $messages = [
        'manualReference.required' => 'A referência é obrigatória',
        'manualReference.min' => 'A referência deve ter pelo menos 5 caracteres',
        'manualReference.max' => 'A referência não pode ter mais de 20 caracteres',
        'manualAmount.required' => 'O valor é obrigatório',
        'manualAmount.min' => 'O valor deve ser maior que 0',
        'manualDate.required' => 'A data é obrigatória',
        'manualDate.date' => 'Data inválida',
    ];

    public function boot(
        PaymentReconciliationService $reconciliationService,
        FeeCalculationService $feeCalculationService
    ) {
        $this->reconciliationService = $reconciliationService;
        $this->feeCalculationService = $feeCalculationService;
    }

    public function mount()
    {
        $this->filterDateFrom = now()->subDays(30)->format('Y-m-d');
        $this->filterDateTo = now()->format('Y-m-d');
        $this->loadStats();
    }

    /* ==================== PROPERTIES SEGURAS ==================== */

    public function getIncomingFilesProperty(): Collection
    {
        if ($this->cachedIncomingFiles !== null) {
            return $this->cachedIncomingFiles;
        }

        try {
            $incomingPath = public_path('bci/bmeps/incoming');
            
            if (!is_dir($incomingPath)) {
                $this->cachedIncomingFiles = collect([]);
                return $this->cachedIncomingFiles;
            }

            $files = glob($incomingPath . '/*.{txt,TXT}', GLOB_BRACE);
            
            if (!is_array($files) || empty($files)) {
                $this->cachedIncomingFiles = collect([]);
                return $this->cachedIncomingFiles;
            }

            $this->cachedIncomingFiles = collect($files)
                ->map(function ($filePath) {
                    try {
                        $fileName = basename($filePath);
                        $fileStats = stat($filePath);
                        
                        if (!$fileStats) {
                            return null;
                        }
                        
                        return [
                            'name' => $fileName,
                            'size' => $fileStats['size'] ?? 0,
                            'size_formatted' => $this->formatFileSize($fileStats['size'] ?? 0),
                            'created_at' => Carbon::createFromTimestamp($fileStats['ctime'] ?? time()),
                            'modified_at' => Carbon::createFromTimestamp($fileStats['mtime'] ?? time()),
                        ];
                    } catch (\Exception $e) {
                        Log::warning('Error reading file stats', ['file' => $filePath ?? 'unknown', 'error' => $e->getMessage()]);
                        return null;
                    }
                })
                ->filter() // Remove nulls
                ->sortByDesc('modified_at')
                ->values(); // Reset keys

        } catch (\Exception $e) {
            Log::error('Error getting incoming files', ['error' => $e->getMessage()]);
            $this->cachedIncomingFiles = collect([]);
        }

        return $this->cachedIncomingFiles;
    }

    public function getUnmatchedPaymentsProperty(): Collection
    {
        if ($this->cachedUnmatchedPayments !== null) {
            return $this->cachedUnmatchedPayments;
        }

        try {
            $payments = DB::table('unmatched_payments')
                ->orderBy('created_at', 'desc')
                ->limit(50)
                ->get();

            if (!$payments) {
                $this->cachedUnmatchedPayments = collect([]);
                return $this->cachedUnmatchedPayments;
            }

            $this->cachedUnmatchedPayments = collect($payments)->map(function ($payment) {
                $payment->can_reconcile_manually = !empty($payment->reference);
                $payment->formatted_amount = 'MT ' . number_format($payment->amount ?? 0, 2);
                $payment->age_days = Carbon::parse($payment->created_at ?? now())->diffInDays(now());
                return $payment;
            });

        } catch (\Exception $e) {
            Log::error('Error getting unmatched payments', ['error' => $e->getMessage()]);
            $this->cachedUnmatchedPayments = collect([]);
        }

        return $this->cachedUnmatchedPayments;
    }

    public function getRecentActivityProperty(): Collection
    {
        if ($this->cachedRecentActivity !== null) {
            return $this->cachedRecentActivity;
        }

        try {
            $activities = collect([]);

            // Logs de reconciliation_logs
            $recentLogs = DB::table('reconciliation_logs')
                ->select([
                    'file_name as description',
                    'status',
                    'started_at as created_at',
                    DB::raw("'file_processing' as type"),
                    'reconciled',
                    'failed'
                ])
                ->orderBy('started_at', 'desc')
                ->limit(5)
                ->get();

            if ($recentLogs) {
                foreach ($recentLogs as $log) {
                    $activities->push([
                        'type' => 'file_processing',
                        'description' => "Ficheiro {$log->description} processado - {$log->reconciled} reconciliados",
                        'created_at' => $log->created_at,
                        'formatted_created_at' => Carbon::parse($log->created_at)->diffForHumans(),
                        'icon' => $this->getActivityIcon('file_processing'),
                        'color' => $this->getStatusColor($log->status),
                    ]);
                }
            }

            // Reconciliações manuais
            $recentManual = DB::table('payment_reconciliations')
                ->where('terminal_id', 'MANUAL')
                ->select([
                    'reference',
                    'amount',
                    'processed_at as created_at'
                ])
                ->orderBy('processed_at', 'desc')
                ->limit(5)
                ->get();

            if ($recentManual) {
                foreach ($recentManual as $manual) {
                    $activities->push([
                        'type' => 'manual_reconciliation',
                        'description' => "Reconciliação manual da referência {$manual->reference} - MT " . number_format($manual->amount ?? 0, 2),
                        'created_at' => $manual->created_at,
                        'formatted_created_at' => Carbon::parse($manual->created_at)->diffForHumans(),
                        'icon' => $this->getActivityIcon('manual_reconciliation'),
                        'color' => 'text-warning',
                    ]);
                }
            }

            $this->cachedRecentActivity = $activities->sortByDesc('created_at')->take(10)->values();

        } catch (\Exception $e) {
            Log::error('Error getting recent activity', ['error' => $e->getMessage()]);
            $this->cachedRecentActivity = collect([]);
        }

        return $this->cachedRecentActivity;
    }

    public function getReconciliationLogsProperty()
    {
        try {
            $query = DB::table('reconciliation_logs')
                ->orderBy('started_at', 'desc');

            if ($this->filterStatus !== 'all') {
                $query->where('status', $this->filterStatus);
            }

            if ($this->filterDateFrom) {
                $query->whereDate('started_at', '>=', $this->filterDateFrom);
            }

            if ($this->filterDateTo) {
                $query->whereDate('started_at', '<=', $this->filterDateTo);
            }

            if ($this->searchTerm) {
                $query->where(function ($q) {
                    $q->where('file_name', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('file_id', 'like', '%' . $this->searchTerm . '%');
                });
            }

            return $query->paginate(20);
        } catch (\Exception $e) {
            Log::error('Error getting reconciliation logs', ['error' => $e->getMessage()]);
            return new \Illuminate\Pagination\LengthAwarePaginator([], 0, 20);
        }
    }

    /**
     * Obtém detalhes das reconciliações com informações dos estudantes
     * Usa fee_assign diretamente pois já tem todos os dados necessários
     */
    public function getDetailedReconciliationsProperty()
    {
        try {
            // Busca direto de fee_assign que tem transaction_id (reconciliados)
            $query = DB::table('fee_assign as fa')
                ->join('users as u', 'fa.student_id', '=', 'u.id')
                ->select([
                    'fa.id',
                    'fa.transaction_id',
                    'fa.reference_number as reference',
                    'fa.student_id',
                    'fa.month as fee_month',
                    'fa.year as fee_year',
                    'fa.amount as base_amount',
                    'fa.fine as fine_amount',
                    'fa.discount as discount_amount',
                    'fa.status',
                    'fa.pay_type as payment_mode',
                    'fa.payment_date',
                    'fa.created_at as processed_at',
                    'u.name as student_name',
                    'u.student_id as student_code',
                ])
                ->whereNotNull('fa.transaction_id') // Só pagamentos com transaction_id (reconciliados)
                ->orderBy('fa.created_at', 'desc');

            // Aplica filtros
            if ($this->filterDateFrom) {
                $query->whereDate('fa.created_at', '>=', $this->filterDateFrom);
            }

            if ($this->filterDateTo) {
                $query->whereDate('fa.created_at', '<=', $this->filterDateTo);
            }

            if ($this->searchTerm) {
                $query->where(function ($q) {
                    $q->where('u.name', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('u.student_id', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('fa.reference_number', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('fa.transaction_id', 'like', '%' . $this->searchTerm . '%');
                });
            }

            $results = $query->paginate(15);

            // Adiciona informação de paid_amount calculado e terminal info
            foreach ($results as $item) {
                $item->paid_amount = ($item->base_amount ?? 0) + ($item->fine_amount ?? 0) - ($item->discount_amount ?? 0);
                // Define terminal_id baseado no tipo de pagamento ou transaction_id
                if ($item->payment_mode === 'Manual' || $item->payment_mode === 'manual') {
                    $item->terminal_id = 'MANUAL';
                    $item->terminal_location = 'Manual Dashboard';
                } elseif (strpos($item->transaction_id, 'MEPS-') === 0 || strpos($item->transaction_id, 'MANUAL-') === 0) {
                    $item->terminal_id = strpos($item->transaction_id, 'MANUAL-') === 0 ? 'MANUAL' : 'MEPS';
                    $item->terminal_location = strpos($item->transaction_id, 'MANUAL-') === 0 ? 'Manual Dashboard' : 'BMEPS/MEPS';
                } else {
                    $item->terminal_id = 'AUTO';
                    $item->terminal_location = 'BMEPS/MEPS';
                }
            }

            // Log para debug
            $sampleResults = [];
            $count = 0;
            foreach ($results as $r) {
                if ($count >= 2) break;
                $sampleResults[] = [
                    'id' => $r->id,
                    'transaction_id' => $r->transaction_id,
                    'reference' => $r->reference,
                    'payment_mode' => $r->payment_mode,
                ];
                $count++;
            }

            Log::info('Detailed reconciliations query executed', [
                'filter_from' => $this->filterDateFrom,
                'filter_to' => $this->filterDateTo,
                'search_term' => $this->searchTerm,
                'results_count' => count($results->items()),
                'total_count' => $results->total(),
                'total_fee_assign_with_transaction' => DB::table('fee_assign')->whereNotNull('transaction_id')->count(),
                'sample_results' => $sampleResults,
            ]);

            return $results;
        } catch (\Exception $e) {
            Log::error('Error getting detailed reconciliations', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return new \Illuminate\Pagination\LengthAwarePaginator([], 0, 15);
        }
    }

    /* ==================== PROCESSAMENTO DE FICHEIROS ==================== */

    public function processUploadedFile()
    {
        $this->validate(['uploadedFile' => 'required|file|mimes:txt,TXT|max:10240']);

        try {
            $this->uploadProgress = 10;

            // Move o arquivo para o diretório de processamento
            $fileName = 'manual_' . now()->format('Ymd_His') . '_' . $this->uploadedFile->getClientOriginalName();
            $incomingPath = public_path('bci/bmeps/incoming');
            
            if (!is_dir($incomingPath)) {
                mkdir($incomingPath, 0775, true);
            }

            $filePath = $incomingPath . '/' . $fileName;
            copy($this->uploadedFile->getRealPath(), $filePath);
            
            $this->uploadProgress = 50;

            // Processa o arquivo
            $results = $this->reconciliationService->processAllPendingFiles();
            
            $this->uploadProgress = 100;

            // Log
            $this->logFileProcessing($fileName, $results);

            $this->dispatchBrowserEvent('file-processed', [
                'message' => 'Ficheiro processado com sucesso!',
                'stats' => $results
            ]);

            $this->reset(['uploadedFile', 'uploadProgress']);
            $this->clearCaches();
            $this->loadStats();

            session()->flash('success',
                "Ficheiro processado: " . ($results['reconciled'] ?? 0) . " pagamentos reconciliados, " .
                ($results['failed'] ?? 0) . " falharam, " . ($results['duplicates'] ?? 0) . " duplicados."
            );

        } catch (\Exception $e) {
            Log::error('Erro no processamento manual de ficheiro', [
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
                'file_name' => $this->uploadedFile ? $this->uploadedFile->getClientOriginalName() : 'unknown'
            ]);

            $this->uploadProgress = 0;
            session()->flash('error', 'Erro ao processar ficheiro: ' . $e->getMessage());
        }
    }

    /* ==================== RECONCILIAÇÃO MANUAL ==================== */

    public function openManualReconciliation()
    {
        $this->reset(['manualReference', 'manualAmount', 'manualDate', 'manualTransactionId', 'manualTerminalId', 'manualNote']);
        $this->manualDate = now()->format('Y-m-d');
        $this->showManualModal = true;
    }

    public function processManualReconciliation()
    {
        \Log::info('processManualReconciliation iniciado', [
            'reference' => $this->manualReference,
            'amount' => $this->manualAmount,
            'date' => $this->manualDate,
            'transaction_id' => $this->manualTransactionId
        ]);

        $this->validate();

        DB::beginTransaction();
        $failedReconciliation = null;
        try {
            // Verifica se existe uma falha registrada para esta referência
            $failedReconciliation = FailedReconciliation::where('reference_number', $this->manualReference)
                ->where('status', 'pending')
                ->first();
            // Verifica se a referência existe
            $paymentReference = PaymentReference::where('reference_number', $this->manualReference)
                ->whereIn('status', ['pending'])
                ->first();

            \Log::info('Referência encontrada', ['payment_reference' => $paymentReference ? $paymentReference->toArray() : null]);

            if (!$paymentReference) {
                throw new \Exception("Referência {$this->manualReference} não encontrada ou já processada.");
            }

            // Busca o estudante
            $student = User::find($paymentReference->student_id);
            if (!$student) {
                throw new \Exception("Estudante não encontrado para a referência {$this->manualReference}.");
            }

            $paymentDate = Carbon::parse($this->manualDate . ' ' . now()->format('H:i:s'));
            
            // Calcula os valores usando o serviço unificado
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $student,
                $paymentReference->fee_month,
                $paymentReference->fee_year,
                $paymentDate,
                $paymentReference
            );

            // Atualiza a PaymentReference
            $paymentReference->update([
                'status' => 'paid',
                'paid_at' => $paymentDate,
                'transaction_id' => $this->manualTransactionId ?: ('MANUAL-' . now()->format('YmdHis') . '-' . substr(md5($this->manualReference), 0, 6)),
                'terminal_id' => $this->manualTerminalId ?: 'MANUAL',
                'fine_amount' => $calculation['fine_amount'],
            ]);

            // Cria o pagamento usando o serviço unificado
            $paymentData = [
                'month' => $paymentReference->fee_month,
                'year' => $paymentReference->fee_year,
                'amount' => (float) $this->manualAmount,
                'fine' => (float) $calculation['fine_amount'],
                'discount' => 0,
                'payment_mode' => 'Manual',
                'paymentMode' => 'Manual',
                'pay_type' => 'manual',
                'transaction_id' => $paymentReference->transaction_id,
                'reference_number' => $this->manualReference,
                'payment_date' => $paymentDate,
                'note' => 'Reconciliação manual: ' . ($this->manualNote ?: 'Processado via dashboard administrativo')
            ];

            $feeAssign = $this->feeCalculationService->createFeePayment(
                $student,
                $paymentData,
                $calculation
            );

            // Registra na tabela de reconciliação
            DB::table('payment_reconciliations')->insert([
                'transaction_id' => $paymentReference->transaction_id,
                'reference' => $this->manualReference,
                'amount' => $this->manualAmount,
                'payment_date' => $paymentDate,
                'terminal_id' => $this->manualTerminalId ?: 'MANUAL',
                'terminal_location' => 'Manual Dashboard',
                'processed_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Log
            Log::info('Manual reconciliation completed', [
                'type' => 'manual_reconciliation',
                'reference' => $this->manualReference,
                'amount' => $this->manualAmount,
                'student_id' => $student->id,
                'transaction_id' => $paymentReference->transaction_id,
                'user_id' => auth()->id(),
                'note' => $this->manualNote,
                'fee_assign_id' => $feeAssign->id,
            ]);

            DB::commit();

            $this->showManualModal = false;
            $this->clearCaches();
            $this->loadStats();

            session()->flash('success',
                "Reconciliação manual concluída: Referência {$this->manualReference}, " .
                "Estudante: {$student->name}, Valor: MT " . number_format($this->manualAmount, 2)
            );

        } catch (\Exception $e) {
            DB::rollback();
            
            Log::error('Manual reconciliation failed', [
                'reference' => $this->manualReference,
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
            ]);

            session()->flash('error', 'Erro na reconciliação manual: ' . $e->getMessage());
        }
    }

    /* ==================== PROCESSAMENTO AUTOMÁTICO ==================== */

    public function runAutomaticReconciliation()
    {
        try {
            $results = $this->reconciliationService->processAllPendingFiles();
            
            // Log
            Log::info('Automatic reconciliation completed', [
                'type' => 'automatic_processing',
                'results' => $results,
                'user_id' => auth()->id(),
                'trigger' => 'manual_dashboard',
            ]);

            $this->clearCaches();
            $this->loadStats();

            if (($results['total_files'] ?? 0) > 0) {
                session()->flash('success',
                    "Processamento automático concluído: " . ($results['total_files'] ?? 0) . " ficheiros, " .
                    ($results['reconciled'] ?? 0) . " reconciliados, " . ($results['failed'] ?? 0) . " falharam."
                );
            } else {
                session()->flash('info', 'Nenhum ficheiro pendente encontrado para processamento.');
            }

        } catch (\Exception $e) {
            Log::error('Automatic reconciliation failed', [
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
            ]);

            session()->flash('error', 'Erro no processamento automático: ' . $e->getMessage());
        }
    }

    /* ==================== GESTÃO DE FICHEIROS ==================== */

    public function deleteIncomingFile($fileName)
    {
        try {
            $filePath = public_path('bci/bmeps/incoming/' . $fileName);
            if (file_exists($filePath)) {
                unlink($filePath);
                
                Log::info('Incoming file deleted manually', [
                    'file_name' => $fileName,
                    'user_id' => auth()->id(),
                ]);

                $this->clearCaches();
                session()->flash('success', "Ficheiro {$fileName} eliminado com sucesso.");
            } else {
                session()->flash('error', "Ficheiro {$fileName} não encontrado.");
            }
        } catch (\Exception $e) {
            Log::error('Failed to delete incoming file', [
                'file_name' => $fileName,
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
            ]);

            session()->flash('error', 'Erro ao eliminar ficheiro: ' . $e->getMessage());
        }
    }

    /* ==================== MÉTODOS AUXILIARES ==================== */

    private function clearCaches()
    {
        $this->cachedIncomingFiles = null;
        $this->cachedUnmatchedPayments = null;
        $this->cachedRecentActivity = null;
        $this->cachedFailedReconciliations = null;
    }

    /* ==================== RECONCILIAÇÕES FALHADAS ==================== */

    public function getFailedReconciliationsProperty(): Collection
    {
        if ($this->cachedFailedReconciliations !== null) {
            return $this->cachedFailedReconciliations;
        }

        try {
            $query = FailedReconciliation::query()
                ->orderBy('created_at', 'desc');

            // Aplica filtros
            if ($this->filterDateFrom) {
                $query->whereDate('created_at', '>=', $this->filterDateFrom);
            }

            if ($this->filterDateTo) {
                $query->whereDate('created_at', '<=', $this->filterDateTo);
            }

            if ($this->searchTerm) {
                $query->where(function ($q) {
                    $q->where('reference_number', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('error_reason', 'like', '%' . $this->searchTerm . '%');
                });
            }

            $failed = $query->where('status', 'pending')->paginate(15);

            $this->cachedFailedReconciliations = collect($failed->items())->map(function ($item) {
                $item->formatted_amount = 'MT ' . number_format($item->amount, 2);
                $item->age_days = Carbon::parse($item->created_at)->diffInDays(now());
                $item->formatted_date = Carbon::parse($item->payment_date)->format('d/m/Y');
                return $item;
            });

        } catch (\Exception $e) {
            Log::error('Error getting failed reconciliations', ['error' => $e->getMessage()]);
            $this->cachedFailedReconciliations = collect([]);
        }

        return $this->cachedFailedReconciliations;
    }

    public function reconcileFromFailed($failedId)
    {
        try {
            $failed = FailedReconciliation::findOrFail($failedId);

            // Preenche o modal de reconciliação manual com dados da falha
            $this->manualReference = $failed->reference_number;
            $this->manualAmount = $failed->amount;
            $this->manualDate = Carbon::parse($failed->payment_date)->format('Y-m-d');
            $this->manualNote = 'Reconciliação de falha: ' . $failed->error_reason;

            $this->showManualModal = true;

        } catch (\Exception $e) {
            Log::error('Error preparing failed reconciliation', [
                'failed_id' => $failedId,
                'error' => $e->getMessage()
            ]);

            session()->flash('error', 'Erro ao preparar reconciliação: ' . $e->getMessage());
        }
    }

    public function ignoreFailedReconciliation($failedId, $reason = null)
    {
        try {
            $failed = FailedReconciliation::findOrFail($failedId);
            $failed->markAsIgnored($reason ?? 'Ignorado manualmente pelo administrador');

            $this->clearCaches();

            session()->flash('success', 'Reconciliação falhada marcada como ignorada.');

            Log::info('Failed reconciliation ignored', [
                'failed_id' => $failedId,
                'reference' => $failed->reference_number,
                'user_id' => auth()->id()
            ]);

        } catch (\Exception $e) {
            Log::error('Error ignoring failed reconciliation', [
                'failed_id' => $failedId,
                'error' => $e->getMessage()
            ]);

            session()->flash('error', 'Erro ao ignorar reconciliação: ' . $e->getMessage());
        }
    }

    public function copyReferenceToClipboard($reference)
    {
        $this->dispatchBrowserEvent('copy-to-clipboard', [
            'text' => $reference,
            'message' => 'Referência copiada para a área de transferência'
        ]);
    }

    /**
     * Obtém dados reais dos últimos 7 dias para o gráfico
     */
    public function getLast7DaysDataProperty()
    {
        try {
            $data = [];

            for ($i = 6; $i >= 0; $i--) {
                $date = now()->subDays($i)->format('Y-m-d');

                // Busca transaction_ids das reconciliações do dia
                $transactionIds = DB::table('payment_reconciliations')
                    ->whereDate('processed_at', $date)
                    ->pluck('transaction_id');

                // Conta pagamentos e calcula total com multas
                $payments = Fee_assign::whereIn('transaction_id', $transactionIds)->get();

                $count = $payments->count();
                $total = $payments->sum(function($payment) {
                    return ($payment->amount ?? 0) + ($payment->fine ?? 0) - ($payment->discount ?? 0);
                });

                $data[] = [
                    'date' => $date,
                    'day_label' => $this->getDayLabel($i),
                    'count' => $count,
                    'total' => round($total, 2),
                ];
            }

            return $data;
        } catch (\Exception $e) {
            Log::error('Error getting last 7 days data', ['error' => $e->getMessage()]);
            return array_fill(0, 7, ['date' => '', 'day_label' => '', 'count' => 0, 'total' => 0]);
        }
    }

    private function getDayLabel($daysAgo)
    {
        $days = ['Dom', 'Seg', 'Ter', 'Qua', 'Qui', 'Sex', 'Sáb'];
        $dayOfWeek = now()->subDays($daysAgo)->dayOfWeek;
        return $days[$dayOfWeek];
    }

    private function loadStats()
    {
        try {
            $todayStart = now()->startOfDay();
            $weekStart = now()->startOfWeek();
            $monthStart = now()->startOfMonth();

            $this->stats = [
                'today' => [
                    'processed_files' => DB::table('reconciliation_logs')->whereDate('started_at', today())->count() ?? 0,
                    'reconciled_payments' => DB::table('payment_reconciliations')->whereDate('processed_at', today())->count() ?? 0,
                    'total_amount' => $this->calculateTotalWithFines(today(), today()),
                    'failed_transactions' => DB::table('reconciliation_logs')->whereDate('started_at', today())->sum('failed') ?? 0,
                ],
                'week' => [
                    'processed_files' => DB::table('reconciliation_logs')->whereBetween('started_at', [$weekStart, now()])->count() ?? 0,
                    'reconciled_payments' => DB::table('payment_reconciliations')->whereBetween('processed_at', [$weekStart, now()])->count() ?? 0,
                    'total_amount' => $this->calculateTotalWithFines($weekStart, now()),
                    'success_rate' => $this->calculateSuccessRate($weekStart, now()),
                ],
                'month' => [
                    'processed_files' => DB::table('reconciliation_logs')->whereBetween('started_at', [$monthStart, now()])->count() ?? 0,
                    'reconciled_payments' => DB::table('payment_reconciliations')->whereBetween('processed_at', [$monthStart, now()])->count() ?? 0,
                    'total_amount' => $this->calculateTotalWithFines($monthStart, now()),
                    'average_processing_time' => $this->calculateAverageProcessingTime($monthStart, now()),
                ],
                'pending' => [
                    'incoming_files' => $this->getIncomingFilesProperty()->count(),
                    'unmatched_payments' => DB::table('unmatched_payments')->where('status', 'unmatched')->count() ?? 0,
                    'pending_references' => PaymentReference::where('status', 'pending')->count() ?? 0,
                    'expired_references' => PaymentReference::where('status', 'pending')->where('expires_at', '<', now())->count() ?? 0,
                    'failed_reconciliations' => FailedReconciliation::where('status', 'pending')->count() ?? 0,
                ],
            ];
        } catch (\Exception $e) {
            Log::error('Error loading stats', ['error' => $e->getMessage()]);
            $this->stats = $this->getDefaultStats();
        }
    }

    private function getDefaultStats(): array
    {
        return [
            'today' => [
                'processed_files' => 0,
                'reconciled_payments' => 0,
                'total_amount' => 0,
                'failed_transactions' => 0,
            ],
            'week' => [
                'processed_files' => 0,
                'reconciled_payments' => 0,
                'total_amount' => 0,
                'success_rate' => 0,
            ],
            'month' => [
                'processed_files' => 0,
                'reconciled_payments' => 0,
                'total_amount' => 0,
                'average_processing_time' => 0,
            ],
            'pending' => [
                'incoming_files' => 0,
                'unmatched_payments' => 0,
                'pending_references' => 0,
                'expired_references' => 0,
                'failed_reconciliations' => 0,
            ],
        ];
    }

    private function calculateSuccessRate($from, $to)
    {
        try {
            $total = DB::table('reconciliation_logs')->whereBetween('started_at', [$from, $to])->sum('total_records') ?? 0;
            $successful = DB::table('reconciliation_logs')->whereBetween('started_at', [$from, $to])->sum('reconciled') ?? 0;
            
            return $total > 0 ? round(($successful / $total) * 100, 1) : 0;
        } catch (\Exception $e) {
            Log::error('Error calculating success rate', ['error' => $e->getMessage()]);
            return 0;
        }
    }

    private function calculateAverageProcessingTime($from, $to)
    {
        try {
            $logs = DB::table('reconciliation_logs')
                ->whereBetween('started_at', [$from, $to])
                ->whereNotNull('completed_at')
                ->select(DB::raw('AVG(TIMESTAMPDIFF(SECOND, started_at, completed_at)) as avg_seconds'))
                ->first();

            return $logs && $logs->avg_seconds ? round($logs->avg_seconds, 1) : 0;
        } catch (\Exception $e) {
            Log::error('Error calculating average processing time', ['error' => $e->getMessage()]);
            return 0;
        }
    }

    /**
     * Calcula o total de pagamentos incluindo multas e descontos
     * Fórmula: VALOR BASE + MULTA - DESCONTO = TOTAL
     */
    private function calculateTotalWithFines($from, $to)
    {
        try {
            // Busca os transaction_ids dos pagamentos reconciliados no período
            $transactionIds = DB::table('payment_reconciliations')
                ->whereBetween('processed_at', [$from, $to])
                ->pluck('transaction_id');

            if ($transactionIds->isEmpty()) {
                return 0;
            }

            // Busca os fee_assign correspondentes e calcula o total correto
            $payments = Fee_assign::whereIn('transaction_id', $transactionIds)->get();

            return $payments->sum(function($payment) {
                return ($payment->amount ?? 0) + ($payment->fine ?? 0) - ($payment->discount ?? 0);
            });
        } catch (\Exception $e) {
            Log::error('Error calculating total with fines', ['error' => $e->getMessage()]);
            return 0;
        }
    }

    private function logFileProcessing($fileName, $results)
    {
        try {
            Log::info('File processing completed via dashboard', [
                'type' => 'file_upload',
                'file_name' => $fileName,
                'results' => $results,
                'user_id' => auth()->id(),
                'ip_address' => request()->ip(),
            ]);
        } catch (\Exception $e) {
            Log::error('Error logging file processing', ['error' => $e->getMessage()]);
        }
    }

    private function getActivityIcon($type)
    {
        $icons = [
            'file_processing' => 'fas fa-upload',
            'manual_reconciliation' => 'fas fa-edit',
            'automatic_processing' => 'fas fa-cogs',
            'error' => 'fas fa-exclamation-triangle',
            'success' => 'fas fa-check-circle',
        ];

        return $icons[$type] ?? 'fas fa-info-circle';
    }

    private function getStatusColor($status)
    {
        $colors = [
            'completed' => 'text-success',
            'failed' => 'text-danger',
            'processing' => 'text-primary',
        ];

        return $colors[$status] ?? 'text-secondary';
    }

    private function formatFileSize($bytes)
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        
        $bytes /= pow(1024, $pow);
        
        return round($bytes, 2) . ' ' . $units[$pow];
    }

    public function render()
    {
        // Garantir que todas as propriedades sejam collections válidas
        $data = [
            'incoming_files' => $this->getIncomingFilesProperty(),
            'unmatched_payments' => $this->getUnmatchedPaymentsProperty(),
            'failed_reconciliations' => $this->getFailedReconciliationsProperty(),
            'recent_activity' => $this->getRecentActivityProperty(),
            'reconciliation_logs' => $this->getReconciliationLogsProperty(),
            'detailed_reconciliations' => $this->getDetailedReconciliationsProperty(),
            'last_7_days_data' => $this->getLast7DaysDataProperty(),
            'stats' => $this->stats,
        ];

        return view('livewire.admin.fee.payment-reconciliation-dashboard', $data);
    }
}
