<?php

namespace App\Http\Livewire\Admin\POS;

use App\Models\POS\Setting;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;

class POSSettings extends Component
{
    use LivewireAlert;

    // Settings
    public $tax_enabled = false;
    public $tax_rate = 16;
    public $store_name = 'COPMOZ Store';
    public $sale_number_prefix = 'POS';
    public $receipt_footer = '';
    public $low_stock_alert = 10;
    public $currency = 'MT';
    public $receipt_format = 'a4';
    public $receipt_printer = false;

    protected $listeners = [
        'confirmedSave',
        'cancelledSave'
    ];

    public function mount()
    {
        $this->loadSettings();
    }

    public function loadSettings()
    {
        $this->tax_enabled = Setting::get('tax_enabled', false);
        $this->tax_rate = Setting::get('tax_rate', 16);
        $this->store_name = Setting::get('store_name', 'COPMOZ Store');
        $this->sale_number_prefix = Setting::get('sale_number_prefix', 'POS');
        $this->receipt_footer = Setting::get('receipt_footer', '');
        $this->low_stock_alert = Setting::get('low_stock_alert', 10);
        $this->currency = Setting::get('currency', 'MT');
        $this->receipt_format = Setting::get('receipt_format', 'a4');
        $this->receipt_printer = Setting::get('receipt_printer', false);
    }

    protected function rules()
    {
        return [
            'store_name' => 'required|string|max:255',
            'sale_number_prefix' => 'required|string|max:10|regex:/^[A-Z]+$/',
            'receipt_footer' => 'nullable|string|max:500',
            'low_stock_alert' => 'required|integer|min:1|max:1000',
            'currency' => 'required|string|max:10',
        ];
    }

    public function save()
    {
        $this->validate();

        $this->confirm('Deseja salvar as configurações?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, salvar',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedSave',
            'onDismissed' => 'cancelledSave'
        ]);
    }

    public function confirmedSave()
    {
        try {
            // Save all settings (tax_rate is managed globally in System Settings)
            Setting::set('tax_enabled', $this->tax_enabled ? 'true' : 'false', 'boolean');
            Setting::set('store_name', $this->store_name, 'string');
            Setting::set('sale_number_prefix', strtoupper($this->sale_number_prefix), 'string');
            Setting::set('receipt_footer', $this->receipt_footer ?? '', 'string');
            Setting::set('low_stock_alert', $this->low_stock_alert, 'number');
            Setting::set('currency', $this->currency, 'string');
            Setting::set('receipt_format', $this->receipt_format, 'string');
            Setting::set('receipt_printer', $this->receipt_printer ? 'true' : 'false', 'boolean');

            $this->alert('success', 'Configurações salvas com sucesso!');
            $this->loadSettings();
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao salvar configurações: ' . $e->getMessage());
        }
    }

    public function cancelledSave()
    {
        $this->alert('info', 'Operação cancelada');
    }

    public function resetToDefaults()
    {
        $this->tax_enabled = false;
        $this->store_name = 'COPMOZ Store';
        $this->sale_number_prefix = 'POS';
        $this->receipt_footer = '';
        $this->low_stock_alert = 10;
        $this->currency = 'MT';
        $this->receipt_printer = false;
        // tax_rate is managed globally in System Settings

        $this->alert('info', 'Valores padrão restaurados. Clique em Salvar para confirmar.');
    }

    public function render()
    {
        return view('livewire.admin.pos.pos-settings');
    }
}
