<?php

namespace App\Http\Livewire\Admin\Renewals;

use App\Models\Classroom;
use App\Models\EnrollmentRenewal;
use App\Models\SystemSetting;
use App\Services\PushNotificationService;
use App\Services\SmsService;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;

class ManageRenewals extends Component
{
    use FindGuard, LivewireAlert;

    public bool $periodOpen = false;
    public string $academicYear = '';
    public string $filterClass = '';
    public string $filterClassroom = '';
    public string $filterStatus = '';

    // Auto-schedule
    public string $autoOpenDate  = ''; // DD/MM
    public string $autoCloseDate = ''; // DD/MM

    // For confirm/cancel actions
    public ?string $actionRenewalId = null;
    public string $pendingAction = ''; // 'confirm' or 'cancel'

    // Notification options (shown when confirming a renewal)
    public bool $notifyPush = true;
    public bool $notifySms  = false;

    // Class transition
    public array  $transitionSelected    = [];
    public string $transitionToClassroom = '';
    public bool   $transitionSelectAll   = false;

    protected $listeners = ['confirmed', 'cancelled'];

    public function updatedFilterClass()
    {
        $this->filterClassroom = '';
    }

    public function mount()
    {
        $this->periodOpen    = (bool) SystemSetting::get('renewal_period_open', false, 'enrollment');
        $this->academicYear  = (string) SystemSetting::get('renewal_academic_year', (string) now()->year, 'enrollment');
        $this->autoOpenDate  = (string) SystemSetting::get('renewal_auto_open_date',  '', 'enrollment');
        $this->autoCloseDate = (string) SystemSetting::get('renewal_auto_close_date', '', 'enrollment');
    }

    public function togglePeriod()
    {
        $this->periodOpen = !$this->periodOpen;
        SystemSetting::set('renewal_period_open', $this->periodOpen ? '1' : '0', 'boolean', 'enrollment', 'Período de renovação aberto');
        SystemSetting::clearGroupCache('enrollment');

        $msg = $this->periodOpen ? 'Período de renovação aberto com sucesso.' : 'Período de renovação fechado.';
        $this->alert('success', $msg);
    }

    public function saveAcademicYear()
    {
        $this->validate(['academicYear' => ['required', 'string', 'max:10']]);
        SystemSetting::set('renewal_academic_year', $this->academicYear, 'string', 'enrollment', 'Ano académico alvo de renovação');
        SystemSetting::clearGroupCache('enrollment');
        $this->alert('success', 'Ano académico atualizado para ' . $this->academicYear);
    }

    public function saveSchedule()
    {
        $this->validate([
            'autoOpenDate'  => ['nullable', 'regex:/^\d{2}\/\d{2}$/'],
            'autoCloseDate' => ['nullable', 'regex:/^\d{2}\/\d{2}$/'],
        ], [
            'autoOpenDate.regex'  => 'Use o formato DD/MM (ex: 01/03)',
            'autoCloseDate.regex' => 'Use o formato DD/MM (ex: 31/03)',
        ]);

        SystemSetting::set('renewal_auto_open_date',  $this->autoOpenDate,  'string', 'enrollment', 'Data automática de abertura do período de renovação (DD/MM)');
        SystemSetting::set('renewal_auto_close_date', $this->autoCloseDate, 'string', 'enrollment', 'Data automática de fecho do período de renovação (DD/MM)');
        SystemSetting::clearGroupCache('enrollment');

        $this->alert('success', 'Datas automáticas guardadas com sucesso.');
    }

    public function clearSchedule()
    {
        $this->autoOpenDate  = '';
        $this->autoCloseDate = '';

        SystemSetting::set('renewal_auto_open_date',  '', 'string', 'enrollment', 'Data automática de abertura do período de renovação (DD/MM)');
        SystemSetting::set('renewal_auto_close_date', '', 'string', 'enrollment', 'Data automática de fecho do período de renovação (DD/MM)');
        SystemSetting::clearGroupCache('enrollment');

        $this->alert('info', 'Datas automáticas removidas. O período será gerido manualmente.');
    }

    public function confirmRenewal(string $renewalId)
    {
        $this->actionRenewalId = $renewalId;
        $this->pendingAction   = 'confirm';
        $this->confirm('Confirmar esta renovação de matrícula?', [
            'toast'             => false,
            'position'          => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, confirmar!',
            'cancelButtonText'  => 'Não, cancelar!',
            'onConfirmed'       => 'confirmed',
            'onDismissed'       => 'cancelled',
        ]);
    }

    public function cancelRenewal(string $renewalId)
    {
        $this->actionRenewalId = $renewalId;
        $this->pendingAction   = 'cancel';
        $this->confirm('Cancelar esta renovação?', [
            'toast'             => false,
            'position'          => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, cancelar renovação!',
            'cancelButtonText'  => 'Não, manter!',
            'onConfirmed'       => 'confirmed',
            'onDismissed'       => 'cancelled',
        ]);
    }

    public function confirmed()
    {
        if (!$this->actionRenewalId) return;

        $renewal = EnrollmentRenewal::find($this->actionRenewalId);
        if (!$renewal) {
            $this->alert('error', 'Renovação não encontrada.');
            return;
        }

        $admin = $this->findGuardType()->user();

        if ($this->pendingAction === 'confirm') {
            $renewal->update([
                'status'       => 'confirmada',
                'confirmed_by' => $admin->id,
                'confirmed_at' => now(),
            ]);

            // Send notifications
            $student = $renewal->student;
            if ($student) {
                $notifyTitle = 'Reserva de Vaga Confirmada';
                $notifyBody  = "Olá {$student->name}, a sua reserva de vaga para o ano {$renewal->academic_year} foi confirmada com sucesso.";

                if ($this->notifyPush) {
                    try {
                        app(PushNotificationService::class)->sendToUsers([$student], $notifyTitle, $notifyBody);
                    } catch (\Exception $e) {
                        \Illuminate\Support\Facades\Log::warning("Push notification failed for renewal {$renewal->id}: " . $e->getMessage());
                    }
                }

                if ($this->notifySms && !empty($student->phone_number)) {
                    try {
                        app(SmsService::class)->send($student->phone_number, $notifyBody);
                    } catch (\Exception $e) {
                        \Illuminate\Support\Facades\Log::warning("SMS notification failed for renewal {$renewal->id}: " . $e->getMessage());
                    }
                }
            }

            $this->alert('success', 'Renovação confirmada com sucesso.');
        } elseif ($this->pendingAction === 'cancel') {
            $renewal->update([
                'status'       => 'cancelada',
                'cancelled_at' => now(),
            ]);
            $this->alert('warning', 'Renovação cancelada.');
        }

        $this->actionRenewalId = null;
        $this->pendingAction   = '';
    }

    public function cancelled()
    {
        $this->actionRenewalId = null;
        $this->pendingAction   = '';
        $this->alert('info', 'Ação cancelada.');
    }

    public function updatedTransitionSelectAll(bool $value)
    {
        if ($value) {
            $year = $this->academicYear ?: now()->year;
            $query = EnrollmentRenewal::confirmada()->notTransitioned()->where('academic_year', $year);
            if ($this->filterClass)    $query->where('from_class', $this->filterClass);
            if ($this->filterClassroom) $query->where('from_classroom_id', $this->filterClassroom);
            $this->transitionSelected = $query->pluck('id')->toArray();
        } else {
            $this->transitionSelected = [];
        }
    }

    public function executeTransition()
    {
        if (empty($this->transitionSelected)) {
            $this->alert('warning', 'Seleccione pelo menos um aluno.');
            return;
        }
        if (!$this->transitionToClassroom) {
            $this->alert('warning', 'Escolha a turma de destino.');
            return;
        }

        $renewals = EnrollmentRenewal::confirmada()
            ->notTransitioned()
            ->whereIn('id', $this->transitionSelected)
            ->get();

        $count = 0;
        foreach ($renewals as $renewal) {
            \App\Models\User::where('id', $renewal->student_id)->update([
                'classroom_id' => $this->transitionToClassroom,
                'year_edu'     => (int) $this->academicYear ?: now()->year,
            ]);
            $renewal->update([
                'to_classroom_id' => $this->transitionToClassroom,
                'transitioned_at' => now(),
            ]);
            $count++;
        }

        $this->transitionSelected    = [];
        $this->transitionToClassroom = '';
        $this->transitionSelectAll   = false;
        $this->alert('success', "{$count} aluno(s) transitado(s) com sucesso para o ano {$this->academicYear}.");
    }

    public function render()
    {
        $year = $this->academicYear ?: now()->year;

        // Per-classroom summary
        $classrooms = Classroom::with(['renewals' => function ($q) use ($year) {
            $q->where('academic_year', $year);
        }])->orderByRaw('CAST(class AS UNSIGNED)')->orderBy('name')->get();

        // Individual renewals for the detail table
        $renewalsQuery = EnrollmentRenewal::with(['student', 'fromClassroom'])
            ->where('academic_year', $year);

        if ($this->filterClass) {
            $renewalsQuery->where('from_class', $this->filterClass);
        }
        if ($this->filterClassroom) {
            $renewalsQuery->where('from_classroom_id', $this->filterClassroom);
        }
        if ($this->filterStatus) {
            $renewalsQuery->where('status', $this->filterStatus);
        }

        $renewals = $renewalsQuery->latest()->get();

        // Distinct classes for filter dropdown
        $classes = Classroom::select('class')->distinct()->orderByRaw('CAST(class AS UNSIGNED)')->pluck('class');

        // Classrooms for turma filter (filtered by selected class if any)
        $filterClassrooms = Classroom::when($this->filterClass, fn($q) => $q->where('class', $this->filterClass))
            ->orderByRaw('CAST(class AS UNSIGNED)')
            ->orderBy('name')
            ->get(['id', 'name', 'class']);

        // Transition: confirmed + not yet transitioned (respects existing filters)
        $transitionQuery = EnrollmentRenewal::with(['student', 'fromClassroom'])
            ->confirmada()
            ->where('academic_year', $year);
        if ($this->filterClass)     $transitionQuery->where('from_class', $this->filterClass);
        if ($this->filterClassroom) $transitionQuery->where('from_classroom_id', $this->filterClassroom);

        $confirmedNotTransitioned = $transitionQuery->notTransitioned()->latest()->get();
        $transitionedCount        = $transitionQuery->clone()->whereNotNull('transitioned_at')->count();

        // Destination classrooms for transition: next class level
        $nextClass  = $this->filterClass ? ((int) $this->filterClass + 1) : null;
        $toClassrooms = Classroom::when($nextClass, fn($q) => $q->where('class', (string) $nextClass))
            ->orderByRaw('CAST(class AS UNSIGNED)')
            ->orderBy('name')
            ->get(['id', 'name', 'class']);

        return view('livewire.admin.renewals.manage-renewals', compact(
            'classrooms', 'renewals', 'classes', 'filterClassrooms',
            'confirmedNotTransitioned', 'transitionedCount', 'toClassrooms'
        ));
    }
}
