<?php

namespace App\Http\Livewire\Admin\Reports;

use App\Services\IvaCalculatorService;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;

class IvaReport extends Component
{
    use FindGuard, LivewireAlert;

    public $startDate;
    public $endDate;
    public $category = null;
    public $reportData = null;
    public $showReport = false;

    protected $rules = [
        'startDate' => 'required|date',
        'endDate' => 'required|date|after_or_equal:startDate',
    ];

    public function mount()
    {
        // Default to current month
        $this->startDate = now()->startOfMonth()->format('Y-m-d');
        $this->endDate = now()->endOfMonth()->format('Y-m-d');
    }

    public function generateReport()
    {
        $this->validate();

        try {
            $service = new IvaCalculatorService();

            // Gerar relatório
            $this->reportData = $service->generateReport(
                $this->startDate,
                $this->endDate,
                $this->category,
                true // autoSync
            );

            $this->showReport = true;
            $this->alert('success', 'Relatório gerado com sucesso!');
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao gerar relatório: ' . $e->getMessage());
        }
    }

    public function syncTransactions()
    {
        try {
            $service = new IvaCalculatorService();
            $result = $service->syncAllTransactions($this->startDate, $this->endDate);

            $message = sprintf(
                'Sincronizado: %d vendas POS, %d pagamentos de fees',
                $result['pos_sales'],
                $result['fee_payments']
            );

            $this->alert('success', $message);

            // Refresh report if it was showing
            if ($this->showReport) {
                $this->generateReport();
            }
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao sincronizar: ' . $e->getMessage());
        }
    }

    public function exportPdf()
    {
        if (!$this->reportData) {
            $this->alert('warning', 'Gere o relatório primeiro!');
            return;
        }

        // TODO: Implementar exportação PDF
        $this->alert('info', 'Exportação PDF em desenvolvimento');
    }

    public function exportExcel()
    {
        if (!$this->reportData) {
            $this->alert('warning', 'Gere o relatório primeiro!');
            return;
        }

        // TODO: Implementar exportação Excel
        $this->alert('info', 'Exportação Excel em desenvolvimento');
    }

    public function render()
    {
        return view('livewire.admin.reports.iva-report');
    }
}
