<?php

namespace App\Http\Livewire\Admin\Sms;

use App\Models\User;
use App\Services\SmsService;
use Carbon\Carbon;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;
use Exception;

class BulkSms extends Component
{
    use LivewireAlert;

    public $studentCode = '';
    public $selectedStudents = [];
    public $message = '';
    public $showTodayStudents = false;
    public $todayStudents = [];
    public $searchResults = [];
    public $sending = false;

    protected $listeners = [
        'confirmSend',
        'cancelSend'
    ];

    protected $rules = [
        'message' => 'required|min:10|max:500',
    ];

    public function mount()
    {
        $this->loadTodayStudents();
    }

    public function loadTodayStudents()
    {
        $this->todayStudents = User::whereDate('created_at', Carbon::today())
            ->where('is_active', 1)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($student) {
                return [
                    'id' => $student->id,
                    'name' => $student->name,
                    'student_id' => $student->student_id,
                    'phone_number' => $student->phone_number,
                    'classroom' => $student->classroom->name ?? 'N/A',
                    'created_at' => $student->created_at->format('H:i'),
                ];
            })
            ->toArray();
    }

    public function updatedStudentCode()
    {
        $this->searchStudent();
    }

    public function searchStudent()
    {
        if (empty($this->studentCode) || strlen($this->studentCode) < 2) {
            $this->searchResults = [];
            return;
        }

        $search = $this->studentCode;

        $this->searchResults = User::where(function($query) use ($search) {
                $query->where('student_id', 'like', '%' . $search . '%')
                      ->orWhere('name', 'like', '%' . $search . '%');
            })
            ->where('is_active', 1)
            ->limit(10)
            ->get()
            ->map(function ($student) {
                return [
                    'id' => $student->id,
                    'name' => $student->name,
                    'student_id' => $student->student_id,
                    'phone_number' => $student->phone_number,
                    'classroom' => $student->classroom->name ?? 'N/A',
                ];
            })
            ->toArray();
    }

    public function addStudent($studentId)
    {
        $student = User::find($studentId);

        if (!$student) {
            $this->alert('error', 'Estudante não encontrado');
            return;
        }

        // Check if already added
        $exists = collect($this->selectedStudents)->contains('id', $studentId);

        if ($exists) {
            $this->alert('warning', 'Estudante já está na lista');
            return;
        }

        $this->selectedStudents[] = [
            'id' => $student->id,
            'name' => $student->name,
            'student_id' => $student->student_id,
            'phone_number' => $student->phone_number,
            'classroom' => $student->classroom->name ?? 'N/A',
        ];

        $this->studentCode = '';
        $this->searchResults = [];

        $this->alert('success', 'Estudante adicionado: ' . $student->name);
    }

    public function addStudentByCode()
    {
        if (empty($this->studentCode)) {
            $this->alert('warning', 'Digite o código do estudante');
            return;
        }

        $student = User::where('student_id', $this->studentCode)->first();

        if (!$student) {
            $this->alert('error', 'Estudante não encontrado com o código: ' . $this->studentCode);
            return;
        }

        $this->addStudent($student->id);
    }

    public function removeStudent($index)
    {
        if (isset($this->selectedStudents[$index])) {
            $name = $this->selectedStudents[$index]['name'];
            unset($this->selectedStudents[$index]);
            $this->selectedStudents = array_values($this->selectedStudents);
            $this->alert('info', 'Removido: ' . $name);
        }
    }

    public function addAllTodayStudents()
    {
        $added = 0;
        foreach ($this->todayStudents as $student) {
            $exists = collect($this->selectedStudents)->contains('id', $student['id']);
            if (!$exists) {
                $this->selectedStudents[] = $student;
                $added++;
            }
        }

        if ($added > 0) {
            $this->alert('success', $added . ' estudante(s) adicionado(s)');
        } else {
            $this->alert('info', 'Todos os estudantes de hoje já estão na lista');
        }
    }

    public function clearSelected()
    {
        $this->selectedStudents = [];
        $this->alert('info', 'Lista limpa');
    }

    public function sendSms()
    {
        $this->validate();

        if (empty($this->selectedStudents)) {
            $this->alert('error', 'Selecione pelo menos um estudante');
            return;
        }

        $this->confirm('Enviar SMS para ' . count($this->selectedStudents) . ' estudante(s)?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, enviar!',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmSend',
            'onDismissed' => 'cancelSend'
        ]);
    }

    public function confirmSend()
    {
        $this->sending = true;
        $smsService = new SmsService();

        $success = 0;
        $failed = 0;
        $errors = [];

        foreach ($this->selectedStudents as $student) {
            try {
                $personalMessage = str_replace(
                    ['{nome}', '{codigo}', '{turma}'],
                    [$student['name'], $student['student_id'], $student['classroom']],
                    $this->message
                );

                $result = $smsService->send($student['phone_number'], $personalMessage);

                if ($result['success']) {
                    $success++;
                } else {
                    $failed++;
                    $errors[] = $student['name'] . ': ' . ($result['message'] ?? 'Erro desconhecido');
                }
            } catch (Exception $e) {
                $failed++;
                $errors[] = $student['name'] . ': ' . $e->getMessage();
            }
        }

        $this->sending = false;

        if ($success > 0 && $failed == 0) {
            $this->alert('success', 'SMS enviado com sucesso para ' . $success . ' estudante(s)');
            $this->selectedStudents = [];
            $this->message = '';
        } elseif ($success > 0 && $failed > 0) {
            $this->alert('warning', 'SMS enviado para ' . $success . ', falhou para ' . $failed . ' estudante(s)');
        } else {
            $this->alert('error', 'Falha ao enviar SMS. Verifique os números de telefone.');
        }
    }

    public function cancelSend()
    {
        $this->alert('info', 'Envio cancelado');
    }

    public function render()
    {
        return view('livewire.admin.sms.bulk-sms');
    }
}
