<?php

namespace App\Http\Livewire\Admin\Students;

use App\Jobs\MailJob;
use App\Models\User;
use App\Models\Admin;
use App\Models\Classroom;
use App\Models\Guardian;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Note\Note;
use Exception;
use App\Services\SmsService;
use App\Services\GoogleVisionOCRService;

class AddStudent extends Component
{
    use FindGuard, LivewireAlert, WithFileUploads;

    public $name, $classroom_id, $sex, $phone_number, $student_id, $class, $email, $dob, $year_edu;
    public $father_name, $mother_name, $birth_place, $birth_district, $birth_province;
    public $document_id, $document_type, $avatar;
    public $guardian_name = '', $guardian_phone = '', $guardian_relationship = 'encarregado';
    public $validatedData;
    public $updateMode = false;

    // Toggle: gerar ID automaticamente ou inserir manualmente
    public $auto_generate_id = true;

    // Upload opcional de BI para preencher automaticamente
    public $bi_document;
    public $processing_bi = false;
    public $bi_processed = false;
    public $extracted_photo_path = null;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function mount()
    {
        $this->year_edu = date('Y');
        $this->student_id = $this->generateStudentId();
    }

    public function updatedAutoGenerateId($value)
    {
        if ($value) {
            $this->student_id = $this->generateStudentId();
        } else {
            $this->student_id = '';
        }
    }

    private function generateStudentId(): string
    {
        $year = (int) date('Y');
        $letter = chr(ord('A') + ($year - 2022));
        $prefix = 'CPM' . $letter;

        $last = User::where('student_id', 'like', "{$prefix}%")
            ->orderBy('student_id', 'desc')
            ->first();

        $seq = 1001;
        if ($last) {
            $num = intval(substr($last->student_id, 4));
            $seq = $num + 1;
        }

        $studentId = $prefix . $seq;

        while (User::where('student_id', $studentId)->exists()) {
            $seq++;
            $studentId = $prefix . $seq;
        }

        return $studentId;
    }

    protected array $rules = [
        'name' => ['required', 'string', 'max:255'],
        'sex' => ['required', 'string', 'max:255'],
        'email' => ['required', 'string', 'max:255', 'unique:users'],
        'student_id' => ['required', 'string', 'max:255', 'unique:users'],
        'classroom_id' => ['required', 'string', 'max:255'],
        'class' => ['required', 'string', 'max:255'],
        'dob' => ['required', 'string', 'max:255'],
        'phone_number' => ['required', 'numeric'],
        'year_edu' => ['required', 'integer', 'min:2020', 'max:2035'],
        'father_name' => ['nullable', 'string', 'max:255'],
        'mother_name' => ['nullable', 'string', 'max:255'],
        'birth_place' => ['nullable', 'string', 'max:255'],
        'birth_district' => ['nullable', 'string', 'max:255'],
        'birth_province' => ['nullable', 'string', 'max:255'],
        'document_id' => ['nullable', 'string', 'max:255'],
        'document_type' => ['nullable', 'string', 'in:BI,Cedula,Passaporte'],
        'avatar' => ['nullable', 'image', 'max:2048'],
        'guardian_name' => ['nullable', 'string', 'max:255'],
        'guardian_phone' => ['nullable', 'string', 'max:20'],
        'guardian_relationship' => ['nullable', 'string', 'in:pai,mae,encarregado,outro'],
    ];

    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);

        // Processar BI automaticamente quando for feito upload
        if ($propertyName === 'bi_document') {
            $this->processBiDocument();
        }
    }

    /**
     * Processar BI com OCR e preencher formulário automaticamente
     */
    public function processBiDocument()
    {
        if (!$this->bi_document) {
            return;
        }

        $this->processing_bi = true;

        try {
            // Salvar temporariamente
            $tempPath = $this->bi_document->store('temp', 'public');
            $fullPath = storage_path('app/public/' . $tempPath);

            // Processar com OCR
            $ocrService = new GoogleVisionOCRService();
            $result = $ocrService->processDocument($fullPath);

            if ($result['success']) {
                // Extrair dados do BI
                $extractedData = $ocrService->extractBIData($result['text']);

                // Preencher campos automaticamente
                if (!empty($extractedData['full_name'])) {
                    $this->name = $extractedData['full_name'];
                }

                if (!empty($extractedData['dob'])) {
                    $this->dob = $extractedData['dob'];
                }

                if (!empty($extractedData['sex'])) {
                    // Converter M/F para Masculino/Feminino
                    $this->sex = $extractedData['sex'] === 'M' ? 'Masculino' :
                                ($extractedData['sex'] === 'F' ? 'Feminino' : $extractedData['sex']);
                }

                if (!empty($extractedData['father_name'])) {
                    $this->father_name = $extractedData['father_name'];
                }

                if (!empty($extractedData['mother_name'])) {
                    $this->mother_name = $extractedData['mother_name'];
                }

                if (!empty($extractedData['birth_place'])) {
                    $this->birth_place = $extractedData['birth_place'];
                }

                if (!empty($extractedData['birth_district'])) {
                    $this->birth_district = $extractedData['birth_district'];
                }

                if (!empty($extractedData['birth_province'])) {
                    $this->birth_province = $extractedData['birth_province'];
                }

                if (!empty($extractedData['document_id'])) {
                    $this->document_id = $extractedData['document_id'];
                }

                if (!empty($extractedData['document_type'])) {
                    $this->document_type = $extractedData['document_type'];
                }

                // Tentar extrair foto do estudante do BI
                $facePath = $ocrService->extractFaceFromDocument($fullPath);
                if ($facePath && file_exists($facePath)) {
                    // Salvar foto extraída no storage
                    $photoContent = file_get_contents($facePath);
                    $photoStoragePath = 'extracted_photos/' . uniqid('photo_') . '.jpg';
                    Storage::disk('public')->put($photoStoragePath, $photoContent);

                    $this->extracted_photo_path = $photoStoragePath;

                    // Limpar arquivo temporário da face
                    @unlink($facePath);
                }

                $this->bi_processed = true;

                if ($this->extracted_photo_path) {
                    $this->alert('success', 'Dados do BI extraídos com sucesso! Foto do estudante detectada automaticamente. Verifique e corrija se necessário.');
                } else {
                    $this->alert('success', 'Dados do BI extraídos com sucesso! Não foi possível detectar a foto - por favor, faça upload manualmente. Verifique e corrija se necessário.');
                }
            } else {
                $this->alert('error', 'Erro ao processar BI: ' . $result['error']);
            }

            // Limpar arquivo temporário
            Storage::disk('public')->delete($tempPath);

        } catch (Exception $e) {
            $this->alert('error', 'Erro ao processar BI: ' . $e->getMessage());
        }

        $this->processing_bi = false;
    }

    public function submit()
    {
        $this->validatedData = $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        // generate password
        $password = rand(100000, 500000);

        // Handle avatar upload - usar avatar padrão se nenhum for fornecido
        $avatarPath = 'avatars/default-avatar.png';
        if ($this->avatar) {
            // Se houver upload manual de avatar, usar esse
            $avatarPath = $this->avatar->store('avatars', 'public');
        } elseif ($this->extracted_photo_path) {
            // Se não houver upload manual mas houver foto extraída do BI, mover para avatars
            $photoContent = Storage::disk('public')->get($this->extracted_photo_path);
            $avatarPath = 'avatars/' . basename($this->extracted_photo_path);
            Storage::disk('public')->put($avatarPath, $photoContent);
            // Limpar foto extraída temporária
            Storage::disk('public')->delete($this->extracted_photo_path);
        }

        $student = User::query()->create([
            'name' => $this->name,
            'sex' => $this->sex,
            'email' => $this->email,
            'student_id' => $this->student_id,
            'classroom_id' => $this->classroom_id,
            'dob' => $this->dob,
            'password' => bcrypt($password),
            'is_paid' => true,
            'phone_number' => $this->phone_number,
            'year_edu' => $this->year_edu,
            'father_name' => $this->father_name,
            'mother_name' => $this->mother_name,
            'birth_place' => $this->birth_place,
            'birth_district' => $this->birth_district,
            'birth_province' => $this->birth_province,
            'document_id' => $this->document_id,
            'document_type' => $this->document_type,
            'avatar' => $avatarPath,
        ]);
        // Auto-create guardian if guardian info provided
        $this->createGuardianForStudent($student);

        // Enviar SMS usando o gateway configurado nas definições
        try {
            $message = "Caro pai, o seu educando " . $this->name . ", foi registado no sistema de gestao escolar COPMOZ, o seu ID do Estudante: $this->student_id e a sua senha: " . $password . "\n\nClique no link para aceder o sistema: https://siga.copmoz.ac.mz";

            $smsService = new SmsService();
            $result = $smsService->send($this->phone_number, $message);

            if (!$result['success']) {
                $this->alert('warning', 'Estudante criado, mas houve erro ao enviar SMS');
            }
        } catch (Exception $e) {
            $this->alert('warning', 'Estudante criado, mas houve erro ao enviar SMS: ' . $e->getMessage());
        }
        Note::createSystemNotification(Admin::class, 'Novo estudante', 'Criado com sucesso ' . $this->name);
        $this->emit('noteAdded');
        $this->alert('success', 'Criado com sucesso ' . $this->name);
        $this->reset();
    }

    private function createGuardianForStudent(User $student): void
    {
        if (empty($this->guardian_phone) || empty($this->guardian_name)) {
            return;
        }

        try {
            $phone = preg_replace('/\D/', '', $this->guardian_phone);
            if (strlen($phone) === 9 && !str_starts_with($phone, '258')) {
                $phone = '258' . $phone;
            }

            $guardian = Guardian::where('phone', $phone)->first();
            $isNew = false;
            $guardianPassword = null;

            if (!$guardian) {
                $guardianPassword = rand(100000, 500000);
                $guardian = Guardian::create([
                    'name'         => $this->guardian_name,
                    'phone'        => $phone,
                    'relationship' => $this->guardian_relationship ?: 'encarregado',
                    'password'     => Hash::make($guardianPassword),
                    'is_active'    => true,
                ]);
                $isNew = true;
            }

            if (!$guardian->students()->where('users.id', $student->id)->exists()) {
                $guardian->students()->attach($student->id, [
                    'id'           => (string) Str::uuid(),
                    'relationship' => $this->guardian_relationship ?: 'encarregado',
                    'is_primary'   => true,
                ]);
            }

            if ($isNew && $guardianPassword) {
                try {
                    $smsService = new SmsService();
                    $guardianMsg = "Caro(a) {$this->guardian_name}, a sua conta no Portal do Encarregado COPMOZ foi criada." .
                        "\nTelefone: {$phone}" .
                        "\nSenha: {$guardianPassword}" .
                        "\n\nAceda: https://siga.copmoz.ac.mz/encarregado";
                    $smsService->send($phone, $guardianMsg);
                } catch (Exception $e) {
                    Log::warning('Failed to send guardian credentials SMS', [
                        'guardian_id' => $guardian->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            Log::info('Guardian linked to student', [
                'guardian_id' => $guardian->id,
                'student_id' => $student->id,
                'is_new_guardian' => $isNew,
            ]);
        } catch (Exception $e) {
            Log::error('Failed to create/link guardian', [
                'error' => $e->getMessage(),
            ]);
        }
    }

    public function cancelled()
    {
        $this->alert('error', 'Voce cancelou.');
    }

    
    public function render()
    {
        return view('livewire.admin.students.add-student', [
            'classes' => Classroom::query()->where(function ($query) {
                $query->orWhere('class', $this->class);
            })->get(),
        ]);
    }
}
