<?php

namespace App\Http\Livewire\Admin\Students;

use App\Models\DocumentScan;
use App\Models\User;
use App\Services\GoogleVisionOCRService;
use Illuminate\Support\Facades\Storage;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;

class DocumentScanner extends Component
{
    use WithFileUploads, LivewireAlert, WithPagination;

    protected $paginationTheme = 'bootstrap';

    // Upload
    public $documents = [];
    public $uploadProgress = 0;

    // Câmera
    public $showCameraModal = false;
    public $capturedPhoto = null;

    // Filtros
    public $filterStatus = 'all';
    public $searchTerm = '';

    // Modal de edição
    public $showEditModal = false;
    public $editingDocument = null;
    public $editData = [];

    // Modal de confirmação
    public $showConfirmModal = false;
    public $confirmingDocument = null;

    // Preview automático (usar IDs em vez de objetos para evitar erro de serialização)
    public $showPreviewModal = false;
    public $previewDocumentId = null;
    public $detectedStudentId = null;

    // Captura de verso do BI
    public $capturingBackSide = false;
    public $backSideDocumentId = null;

    protected $listeners = [
        'refreshDocuments' => '$refresh',
    ];

    public function mount()
    {
        //
    }

    /**
     * Fechar modal da câmera e resetar estados
     */
    public function closeCameraModal()
    {
        $this->showCameraModal = false;
        $this->capturedPhoto = null;
        $this->capturingBackSide = false;
        $this->backSideDocumentId = null;
    }

    /**
     * Propriedades computadas para buscar objetos por ID
     */
    public function getPreviewDocumentProperty()
    {
        return $this->previewDocumentId ? DocumentScan::find($this->previewDocumentId) : null;
    }

    public function getDetectedStudentProperty()
    {
        return $this->detectedStudentId ? User::find($this->detectedStudentId) : null;
    }

    /**
     * Upload e processamento automático de documentos
     */
    public function uploadDocuments()
    {
        $this->validate([
            'documents.*' => 'required|file|mimes:jpeg,jpg,png,gif,pdf|max:10240', // máx 10MB
        ]);

        try {
            $ocrService = new GoogleVisionOCRService();

            // Processar primeiro documento e mostrar preview
            if (count($this->documents) > 0) {
                $document = $this->documents[0];

                // Salvar arquivo (imagem ou PDF)
                $filename = time() . '_' . $document->getClientOriginalName();
                $path = $document->storeAs('document-scans', $filename, 'public');

                // Criar registro
                $scan = DocumentScan::create([
                    'image_path' => $path,
                    'image_filename' => $filename,
                    'image_size' => $document->getSize(),
                    'status' => 'processing',
                    'uploaded_by' => auth()->id(),
                ]);

                // Processar com OCR (detecta automaticamente se é imagem ou PDF)
                $fullPath = storage_path('app/public/' . $path);
                $result = $ocrService->processDocument($fullPath);

                if ($result['success']) {
                    // Extrair dados do BI
                    $extractedData = $ocrService->extractBIData($result['text']);

                    // Atualizar registro
                    $scan->update([
                        'status' => 'processed',
                        'raw_ocr_text' => $result['text'],
                        'ocr_data' => $result['annotations'] ?? null,
                        'confidence_score' => $result['confidence'],
                        'document_type' => $extractedData['document_type'],
                        'document_id' => $extractedData['document_id'],
                        'full_name' => $extractedData['full_name'],
                        'dob' => $extractedData['dob'],
                        'father_name' => $extractedData['father_name'],
                        'mother_name' => $extractedData['mother_name'],
                        'birth_place' => $extractedData['birth_place'],
                        'birth_district' => $extractedData['birth_district'],
                        'birth_province' => $extractedData['birth_province'],
                        'sex' => $extractedData['sex'],
                        'requires_review' => $result['confidence'] < 90,
                    ]);

                    // Detectar estudante automaticamente usando LIKE %
                    $detectedStudent = $this->detectStudent($extractedData['full_name']);

                    if ($detectedStudent) {
                        $scan->update([
                            'student_id' => $detectedStudent->id,
                            'student_code' => $detectedStudent->student_id,
                        ]);
                    }

                    // Mostrar preview automático (armazenar IDs, não objetos)
                    $this->previewDocumentId = $scan->id;
                    $this->detectedStudentId = $detectedStudent ? $detectedStudent->id : null;
                    $this->showPreviewModal = true;

                } else {
                    $scan->markAsFailed($result['error']);
                    $this->alert('error', 'Erro no OCR: ' . $result['error']);
                }
            }

            // Limpar apenas o primeiro documento processado
            array_shift($this->documents);

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao processar documento: ' . $e->getMessage());
        }
    }

    /**
     * Processar foto capturada pela câmera
     */
    public function processCapturedPhoto()
    {
        // Se está capturando verso, usar método específico
        if ($this->capturingBackSide) {
            return $this->processBackSidePhoto();
        }

        if (!$this->capturedPhoto) {
            $this->alert('error', 'Nenhuma foto capturada');
            return;
        }

        try {
            $ocrService = new GoogleVisionOCRService();

            // Decodificar imagem base64
            $imageData = $this->capturedPhoto;

            // Remover prefixo data:image/png;base64,
            if (strpos($imageData, 'data:image') === 0) {
                $imageData = substr($imageData, strpos($imageData, ',') + 1);
            }

            $imageData = base64_decode($imageData);

            // Salvar arquivo temporariamente
            $filename = time() . '_camera_capture.jpg';
            $path = 'document-scans/' . $filename;
            Storage::disk('public')->put($path, $imageData);

            // Criar registro
            $scan = DocumentScan::create([
                'image_path' => $path,
                'image_filename' => $filename,
                'image_size' => strlen($imageData),
                'status' => 'processing',
                'uploaded_by' => auth()->id(),
            ]);

            // Processar com OCR
            $fullPath = storage_path('app/public/' . $path);
            $result = $ocrService->processDocument($fullPath);

            if ($result['success']) {
                // Extrair dados do BI
                $extractedData = $ocrService->extractBIData($result['text']);

                // Atualizar registro
                $scan->update([
                    'status' => 'processed',
                    'raw_ocr_text' => $result['text'],
                    'ocr_data' => $result['annotations'] ?? null,
                    'confidence_score' => $result['confidence'],
                    'document_type' => $extractedData['document_type'],
                    'document_id' => $extractedData['document_id'],
                    'full_name' => $extractedData['full_name'],
                    'dob' => $extractedData['dob'],
                    'father_name' => $extractedData['father_name'],
                    'mother_name' => $extractedData['mother_name'],
                    'birth_place' => $extractedData['birth_place'],
                    'birth_district' => $extractedData['birth_district'],
                    'birth_province' => $extractedData['birth_province'],
                    'sex' => $extractedData['sex'],
                    'requires_review' => $result['confidence'] < 90,
                ]);

                // Detectar estudante automaticamente
                $detectedStudent = $this->detectStudent($extractedData['full_name']);

                if ($detectedStudent) {
                    $scan->update([
                        'student_id' => $detectedStudent->id,
                        'student_code' => $detectedStudent->student_id,
                    ]);
                }

                // Fechar modal de câmera e mostrar preview
                $this->showCameraModal = false;
                $this->capturedPhoto = null;
                $this->previewDocumentId = $scan->id;
                $this->detectedStudentId = $detectedStudent ? $detectedStudent->id : null;
                $this->showPreviewModal = true;

            } else {
                $scan->markAsFailed($result['error']);
                $this->alert('error', 'Erro no OCR: ' . $result['error']);
            }

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao processar foto: ' . $e->getMessage());
        }
    }

    /**
     * Iniciar captura do verso do BI
     */
    public function startCapturingBackSide($documentId)
    {
        $this->backSideDocumentId = $documentId;
        $this->capturingBackSide = true;
        $this->showPreviewModal = false;
        $this->showCameraModal = true;
    }

    /**
     * Processar foto do verso capturada pela câmera
     */
    public function processBackSidePhoto()
    {
        if (!$this->capturedPhoto || !$this->backSideDocumentId) {
            $this->alert('error', 'Erro ao processar verso do documento');
            return;
        }

        try {
            $scan = DocumentScan::find($this->backSideDocumentId);

            if (!$scan) {
                $this->alert('error', 'Documento não encontrado');
                return;
            }

            $ocrService = new GoogleVisionOCRService();

            // Decodificar imagem base64
            $imageData = $this->capturedPhoto;

            // Remover prefixo data:image/png;base64,
            if (strpos($imageData, 'data:image') === 0) {
                $imageData = substr($imageData, strpos($imageData, ',') + 1);
            }

            $imageData = base64_decode($imageData);

            // Salvar arquivo do verso
            $filename = time() . '_back_' . $scan->image_filename;
            $path = 'document-scans/' . $filename;
            Storage::disk('public')->put($path, $imageData);

            // Processar com OCR
            $fullPath = storage_path('app/public/' . $path);
            $result = $ocrService->processDocument($fullPath);

            if ($result['success']) {
                // Extrair dados do verso (pai, mãe, etc)
                $extractedData = $ocrService->extractBIData($result['text']);

                // Atualizar registro com dados do verso + dados extraídos
                $scan->update([
                    'document_side' => 'ambos',
                    'back_image_path' => $path,
                    'back_image_filename' => $filename,
                    'back_image_size' => strlen($imageData),
                    'back_raw_ocr_text' => $result['text'],
                    'back_ocr_data' => $result['annotations'] ?? null,
                    // Atualizar pai e mãe extraídos do verso
                    'father_name' => $extractedData['father_name'] ?? $scan->father_name,
                    'mother_name' => $extractedData['mother_name'] ?? $scan->mother_name,
                ]);

                $this->alert('success', 'Verso do BI processado com sucesso!');

                // Fechar modal de câmera e voltar ao preview
                $this->showCameraModal = false;
                $this->capturedPhoto = null;
                $this->capturingBackSide = false;
                $this->previewDocumentId = $scan->id;
                $this->showPreviewModal = true;

            } else {
                $this->alert('error', 'Erro no OCR do verso: ' . $result['error']);
            }

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao processar verso: ' . $e->getMessage());
        }
    }

    /**
     * Mostrar documento processado via câmera (chamado após upload AJAX)
     */
    public function showProcessedDocument($documentId)
    {
        $scan = DocumentScan::find($documentId);

        if (!$scan) {
            $this->alert('error', 'Documento não encontrado');
            return;
        }

        // Detectar estudante
        $detectedStudent = $this->detectStudent($scan->full_name);

        if ($detectedStudent) {
            $scan->update([
                'student_id' => $detectedStudent->id,
                'student_code' => $detectedStudent->student_id,
            ]);
        }

        // Fechar modal de câmera e mostrar preview
        $this->showCameraModal = false;
        $this->capturedPhoto = null;
        $this->capturingBackSide = false;
        $this->backSideDocumentId = null;
        $this->previewDocumentId = $scan->id;
        $this->detectedStudentId = $detectedStudent ? $detectedStudent->id : null;
        $this->showPreviewModal = true;
    }

    /**
     * Detectar estudante automaticamente usando LIKE %
     */
    protected function detectStudent($fullName)
    {
        if (empty($fullName)) {
            return null;
        }

        // Limpar e preparar nome para busca
        $cleanName = trim($fullName);

        // Buscar estudante usando LIKE % no nome
        // Primeiro tenta busca exata
        $student = User::where('name', 'LIKE', "%{$cleanName}%")
            ->first();

        // Se não encontrou, tenta por partes do nome (sobrenome)
        if (!$student) {
            $nameParts = explode(' ', $cleanName);
            if (count($nameParts) >= 2) {
                // Pegar primeiro e último nome
                $firstName = $nameParts[0];
                $lastName = end($nameParts);

                $student = User::where('name', 'LIKE', "%{$firstName}%")
                    ->where('name', 'LIKE', "%{$lastName}%")
                    ->first();
            }
        }

        return $student;
    }

    /**
     * Editar dados extraídos
     */
    public function editDocument($id)
    {
        $this->editingDocument = DocumentScan::findOrFail($id);
        $this->editData = [
            'student_code' => $this->editingDocument->student_code,
            'document_id' => $this->editingDocument->document_id,
            'document_type' => $this->editingDocument->document_type ?? 'BI',
            'full_name' => $this->editingDocument->full_name,
            'dob' => $this->editingDocument->dob?->format('Y-m-d'),
            'father_name' => $this->editingDocument->father_name,
            'mother_name' => $this->editingDocument->mother_name,
            'birth_place' => $this->editingDocument->birth_place,
            'birth_district' => $this->editingDocument->birth_district,
            'birth_province' => $this->editingDocument->birth_province,
            'sex' => $this->editingDocument->sex,
        ];
        $this->showEditModal = true;
    }

    /**
     * Salvar edições
     */
    public function saveEdit()
    {
        $this->validate([
            'editData.full_name' => 'required|string',
            'editData.dob' => 'nullable|date',
            'editData.document_id' => 'nullable|string',
        ]);

        $this->editingDocument->update($this->editData);
        $this->showEditModal = false;
        $this->alert('success', 'Dados atualizados com sucesso!');
    }

    /**
     * Confirmar e aplicar ao estudante
     */
    public function confirmDocument($id)
    {
        $this->confirmingDocument = DocumentScan::findOrFail($id);
        $this->showConfirmModal = true;
    }

    /**
     * Confirmar preview e aplicar dados ao estudante
     */
    public function confirmPreview()
    {
        try {
            \Log::info('=== CONFIRMANDO PREVIEW ===');
            \Log::info('Preview Document ID: ' . $this->previewDocumentId);
            \Log::info('Detected Student ID: ' . $this->detectedStudentId);

            // Buscar objetos pelos IDs
            $scan = DocumentScan::find($this->previewDocumentId);
            $student = $this->detectedStudentId ? User::find($this->detectedStudentId) : null;

            \Log::info('Scan encontrado: ' . ($scan ? 'SIM (ID: ' . $scan->id . ')' : 'NÃO'));
            \Log::info('Student encontrado: ' . ($student ? 'SIM (ID: ' . $student->id . ', Nome: ' . $student->name . ')' : 'NÃO'));

            if (!$scan) {
                $this->alert('error', 'Documento não encontrado');
                return;
            }

            if ($student) {
                \Log::info('Dados a atualizar:');
                \Log::info('  DOB: ' . $scan->dob);
                \Log::info('  Pai: ' . $scan->father_name);
                \Log::info('  Mãe: ' . $scan->mother_name);
                \Log::info('  Local Nasc: ' . $scan->birth_place);
                \Log::info('  Distrito: ' . $scan->birth_district);
                \Log::info('  Província: ' . $scan->birth_province);
                \Log::info('  BI: ' . $scan->document_id);
                \Log::info('  Sexo: ' . $scan->sex);

                // Preparar data sem hora (apenas Y-m-d)
                $dob = $scan->dob;
                if ($dob && strlen($dob) > 10) {
                    $dob = substr($dob, 0, 10); // Pegar apenas YYYY-MM-DD
                }

                // Converter sexo para formato completo
                $sex = $scan->sex;
                if ($sex === 'M') {
                    $sex = 'Masculino';
                } elseif ($sex === 'F') {
                    $sex = 'Feminino';
                }

                // Atualizar dados do estudante
                $updated = $student->update([
                    'name' => $scan->full_name, // Atualizar nome completo
                    'dob' => $dob, // Data sem hora
                    'father_name' => $scan->father_name,
                    'mother_name' => $scan->mother_name,
                    'birth_place' => $scan->birth_place,
                    'birth_district' => $scan->birth_district,
                    'birth_province' => $scan->birth_province,
                    'document_id' => $scan->document_id,
                    'document_type' => $scan->document_type,
                    'sex' => $sex, // Masculino ou Feminino
                ]);

                \Log::info('Update retornou: ' . ($updated ? 'true' : 'false'));
                \Log::info('Dados após update:');
                $student->refresh();
                \Log::info('  Nome: ' . $student->name);
                \Log::info('  DOB: ' . $student->dob);
                \Log::info('  Sexo: ' . $student->sex);
                \Log::info('  Pai: ' . $student->father_name);
                \Log::info('  Mãe: ' . $student->mother_name);
                \Log::info('  BI: ' . $student->document_id);

                // Marcar como confirmado
                $scan->markAsConfirmed();

                $this->alert('success', 'Dados aplicados a ' . $student->name . ' com sucesso!');
            } else {
                // Se não detectou estudante, apenas marcar como processado
                $scan->markAsProcessed();
                $this->alert('warning', 'Documento processado, mas estudante não foi detectado automaticamente');
            }

            // Fechar modal
            $this->showPreviewModal = false;
            $this->previewDocumentId = null;
            $this->detectedStudentId = null;

            // Se ainda há documentos na fila, processar próximo
            if (count($this->documents) > 0) {
                $this->uploadDocuments();
            } else {
                $this->emit('refreshDocuments');
            }

        } catch (\Exception $e) {
            \Log::error('ERRO ao aplicar dados: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            $this->alert('error', 'Erro ao aplicar dados: ' . $e->getMessage());
        }
    }

    /**
     * Pular documento atual
     */
    public function skipPreview()
    {
        // Marcar como requer revisão
        if ($this->previewDocumentId) {
            $scan = DocumentScan::find($this->previewDocumentId);
            if ($scan) {
                $scan->update(['requires_review' => true]);
            }
        }

        // Fechar modal
        $this->showPreviewModal = false;
        $this->previewDocumentId = null;
        $this->detectedStudentId = null;

        // Se ainda há documentos na fila, processar próximo
        if (count($this->documents) > 0) {
            $this->uploadDocuments();
        } else {
            $this->emit('refreshDocuments');
        }
    }

    /**
     * Aplicar dados ao estudante (método legado para modal de confirmação manual)
     */
    public function applyToStudent()
    {
        try {
            $scan = $this->confirmingDocument;

            // Buscar estudante pelo código
            if ($scan->student_code) {
                $student = User::where('student_id', $scan->student_code)->first();

                if ($student) {
                    // Atualizar dados do estudante
                    $student->update([
                        'dob' => $scan->dob,
                        'father_name' => $scan->father_name,
                        'mother_name' => $scan->mother_name,
                        'birth_place' => $scan->birth_place,
                        'birth_district' => $scan->birth_district,
                        'birth_province' => $scan->birth_province,
                        'document_id' => $scan->document_id,
                        'document_type' => $scan->document_type,
                        'sex' => $scan->sex,
                    ]);

                    // Marcar como confirmado
                    $scan->update([
                        'student_id' => $student->id,
                    ]);
                    $scan->markAsConfirmed();

                    $this->showConfirmModal = false;
                    $this->alert('success', 'Dados aplicados ao estudante com sucesso!');
                } else {
                    $this->alert('error', 'Estudante não encontrado com código: ' . $scan->student_code);
                }
            } else {
                $this->alert('error', 'Código do estudante não informado');
            }

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao aplicar dados: ' . $e->getMessage());
        }
    }

    /**
     * Deletar documento
     */
    public function deleteDocument($id)
    {
        try {
            $scan = DocumentScan::findOrFail($id);

            // Deletar imagem
            if (Storage::disk('public')->exists($scan->image_path)) {
                Storage::disk('public')->delete($scan->image_path);
            }

            $scan->delete();
            $this->alert('success', 'Documento deletado com sucesso!');

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao deletar documento: ' . $e->getMessage());
        }
    }

    /**
     * Reprocessar documento
     */
    public function reprocessDocument($id)
    {
        try {
            $scan = DocumentScan::findOrFail($id);
            $ocrService = new GoogleVisionOCRService();

            $scan->update(['status' => 'processing']);

            $fullPath = storage_path('app/public/' . $scan->image_path);
            $result = $ocrService->processDocument($fullPath);

            if ($result['success']) {
                $extractedData = $ocrService->extractBIData($result['text']);

                $scan->update([
                    'status' => 'processed',
                    'raw_ocr_text' => $result['text'],
                    'confidence_score' => $result['confidence'],
                    'document_type' => $extractedData['document_type'],
                    'document_id' => $extractedData['document_id'],
                    'full_name' => $extractedData['full_name'],
                    'dob' => $extractedData['dob'],
                    'father_name' => $extractedData['father_name'],
                    'mother_name' => $extractedData['mother_name'],
                    'birth_place' => $extractedData['birth_place'],
                    'birth_district' => $extractedData['birth_district'],
                    'birth_province' => $extractedData['birth_province'],
                    'sex' => $extractedData['sex'],
                    'requires_review' => $result['confidence'] < 90,
                    'error_message' => null,
                ]);

                $this->alert('success', 'Documento reprocessado com sucesso!');
            } else {
                $scan->markAsFailed($result['error']);
                $this->alert('error', 'Erro no reprocessamento: ' . $result['error']);
            }

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao reprocessar: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $query = DocumentScan::query()
            ->with(['student', 'uploader'])
            ->orderBy('created_at', 'desc');

        // Filtros
        if ($this->filterStatus !== 'all') {
            $query->where('status', $this->filterStatus);
        }

        if ($this->searchTerm) {
            $query->where(function ($q) {
                $q->where('full_name', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('document_id', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('student_code', 'like', '%' . $this->searchTerm . '%');
            });
        }

        $scannedDocuments = $query->paginate(15);

        // Estatísticas
        $stats = [
            'total' => DocumentScan::count(),
            'pending' => DocumentScan::where('status', 'pending')->count(),
            'processed' => DocumentScan::where('status', 'processed')->count(),
            'confirmed' => DocumentScan::where('status', 'confirmed')->count(),
            'requires_review' => DocumentScan::where('requires_review', true)->count(),
        ];

        return view('livewire.admin.students.document-scanner', compact('scannedDocuments', 'stats'));
    }
}
