<?php

namespace App\Http\Livewire\User\Renewal;

use App\Models\Classroom;
use App\Models\EnrollmentRenewal;
use App\Models\SystemSetting;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;

class RenewEnrollment extends Component
{
    use FindGuard, LivewireAlert;

    protected $listeners = ['confirmed', 'cancelled'];

    public function submit()
    {
        $user = $this->findGuardType()->user();

        $periodOpen   = (bool) SystemSetting::get('renewal_period_open', false, 'enrollment');
        $academicYear = (string) SystemSetting::get('renewal_academic_year', (string) now()->year, 'enrollment');

        if (!$periodOpen) {
            $this->alert('error', 'O período de renovação não está aberto.');
            return;
        }

        $existing = EnrollmentRenewal::where('student_id', $user->id)
            ->where('academic_year', $academicYear)
            ->first();

        if ($existing) {
            $this->alert('warning', 'Já submeteu uma renovação para o ano ' . $academicYear . '.');
            return;
        }

        $this->confirm('Confirmar reserva de vaga para ' . $academicYear . '?', [
            'toast'             => false,
            'position'          => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, confirmar!',
            'cancelButtonText'  => 'Não, cancelar!',
            'onConfirmed'       => 'confirmed',
            'onDismissed'       => 'cancelled',
        ]);
    }

    public function confirmed()
    {
        $user = $this->findGuardType()->user();

        $periodOpen   = (bool) SystemSetting::get('renewal_period_open', false, 'enrollment');
        $academicYear = (string) SystemSetting::get('renewal_academic_year', (string) now()->year, 'enrollment');

        if (!$periodOpen) {
            $this->alert('error', 'O período de renovação foi fechado entretanto.');
            return redirect()->route('user.home');
        }

        $existing = EnrollmentRenewal::where('student_id', $user->id)
            ->where('academic_year', $academicYear)
            ->first();

        if ($existing) {
            $this->alert('warning', 'Já tem uma renovação para o ano ' . $academicYear . '.');
            return redirect()->route('user.home');
        }

        $classroom  = Classroom::find($user->classroom_id);
        $fromClass  = $classroom ? (string) $classroom->class : '';
        $toClass    = $fromClass ? (string) ((int) $fromClass + 1) : '';

        EnrollmentRenewal::create([
            'student_id'        => $user->id,
            'from_classroom_id' => $user->classroom_id,
            'from_class'        => $fromClass,
            'to_class'          => $toClass,
            'academic_year'     => $academicYear,
            'status'            => 'pendente',
        ]);

        $this->alert('success', 'Reserva submetida com sucesso! Aguarde confirmação da secretaria.');
        return redirect()->route('user.home');
    }

    public function cancelled()
    {
        $this->alert('info', 'Reserva cancelada.');
    }

    public function render()
    {
        $user = $this->findGuardType()->user();

        $academicYear = (string) SystemSetting::get('renewal_academic_year', (string) now()->year, 'enrollment');
        $classroom    = Classroom::find($user->classroom_id);
        $fromClass    = $classroom ? (string) $classroom->class : '—';
        $toClass      = is_numeric($fromClass) ? (string) ((int) $fromClass + 1) : '—';

        $existingRenewal = EnrollmentRenewal::where('student_id', $user->id)
            ->where('academic_year', $academicYear)
            ->first();

        return view('livewire.user.renewal.renew-enrollment', compact(
            'user', 'academicYear', 'classroom', 'fromClass', 'toClass', 'existingRenewal'
        ));
    }
}
