<?php

namespace App\Models;

use App\Traits\Uuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class DocumentClassConfiguration extends Model
{
    use HasFactory, Uuids;

    public $incrementing = false;
    protected $keyType = 'string';

    protected $table = 'document_class_configurations';

    protected $fillable = [
        'class_level',
        'document_type',
        'academic_year',
        'document_title',
        'description',
        'is_active',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'class_level' => 'integer',
        'academic_year' => 'integer',
        'is_active' => 'boolean',
    ];

    /**
     * Document types constants
     */
    public const TYPE_CERTIFICATE = 'certificate';
    public const TYPE_DECLARATION = 'declaration';

    /**
     * Get available document types
     */
    public static function getDocumentTypes(): array
    {
        return [
            self::TYPE_CERTIFICATE => 'Certificado',
            self::TYPE_DECLARATION => 'Declaração',
        ];
    }

    /**
     * Relationships
     */
    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    public function updatedBy(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'updated_by');
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeForClass($query, int $classLevel)
    {
        return $query->where('class_level', $classLevel);
    }

    public function scopeForYear($query, ?int $year)
    {
        return $query->where(function ($q) use ($year) {
            $q->where('academic_year', $year)
              ->orWhereNull('academic_year');
        });
    }

    public function scopeCertificates($query)
    {
        return $query->where('document_type', self::TYPE_CERTIFICATE);
    }

    public function scopeDeclarations($query)
    {
        return $query->where('document_type', self::TYPE_DECLARATION);
    }

    /**
     * Get configuration for a specific class level and year
     */
    public static function getForClass(int $classLevel, ?int $academicYear = null): ?self
    {
        $year = $academicYear ?? (int) date('Y');

        // First try to find year-specific config
        $config = self::active()
            ->forClass($classLevel)
            ->where('academic_year', $year)
            ->first();

        // If not found, try to find generic config (null year)
        if (!$config) {
            $config = self::active()
                ->forClass($classLevel)
                ->whereNull('academic_year')
                ->first();
        }

        return $config;
    }

    /**
     * Get all configured class levels
     */
    public static function getConfiguredClasses(?int $academicYear = null): array
    {
        $query = self::active();

        if ($academicYear) {
            $query->where(function ($q) use ($academicYear) {
                $q->where('academic_year', $academicYear)
                  ->orWhereNull('academic_year');
            });
        }

        return $query->orderBy('class_level')
            ->pluck('class_level')
            ->unique()
            ->values()
            ->toArray();
    }

    /**
     * Check if class level is configured
     */
    public static function isClassConfigured(int $classLevel, ?int $academicYear = null): bool
    {
        return self::getForClass($classLevel, $academicYear) !== null;
    }

    /**
     * Helpers
     */
    public function isCertificate(): bool
    {
        return $this->document_type === self::TYPE_CERTIFICATE;
    }

    public function isDeclaration(): bool
    {
        return $this->document_type === self::TYPE_DECLARATION;
    }

    public function getDocumentTypeLabel(): string
    {
        return self::getDocumentTypes()[$this->document_type] ?? 'Desconhecido';
    }

    public function getDocumentTypeBadge(): string
    {
        return match($this->document_type) {
            self::TYPE_CERTIFICATE => '<span class="badge bg-success">Certificado</span>',
            self::TYPE_DECLARATION => '<span class="badge bg-info">Declaração</span>',
            default => '<span class="badge bg-secondary">Desconhecido</span>',
        };
    }
}
