<?php

namespace App\Services;

use App\Models\User;
use App\Models\FeeStructure;
use App\Models\PaymentReference;
use App\Models\Fee_assign;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class FeeCalculationService
{
    /**
     * Calcula taxa base, multa e total para um estudante baseado na classe e data
     */
    public function calculateFeeForStudent(
        User $student,
        ?string $month = null,
        ?int $year = null,
        ?Carbon $paymentDate = null,
        ?PaymentReference $paymentReference = null
    ): array {
        $paymentDate = $paymentDate ?? Carbon::now();
        $month = $month ?? $this->monthInPortuguese($paymentDate->month);
        $year = $year ?? $paymentDate->year;
        
        // Busca a estrutura de taxa do estudante baseada na sua classe
        $studentClassroom = $student->class->class ?? $student->classroom->class ?? null;
        
        if (!$studentClassroom) {
            Log::warning('Student has no classroom assigned', ['student_id' => $student->id]);
            return $this->defaultCalculation(0, 0);
        }

        // Busca estruturas de taxa aplicáveis
        $applicableFeeStructures = $this->getApplicableFeeStructures($studentClassroom, $month);

        if ($applicableFeeStructures->isEmpty()) {
            // Log removido para evitar excesso de entradas no laravel.log
            return $this->defaultCalculation(0, 0);
        }

        // Calcula taxa base
        $baseFee = $applicableFeeStructures->sum('monthly_fee');

        // Calcula multa baseada na data de vencimento
        $fineCalculation = $this->calculateFine(
            $applicableFeeStructures,
            $baseFee,
            $month,
            $year,
            $paymentDate,
            $paymentReference
        );

        $totalAmount = $baseFee + $fineCalculation['fine_amount'];

        // Log removido para evitar excesso de entradas no laravel.log

        return [
            'base_amount' => round($baseFee, 2),
            'fine_amount' => round($fineCalculation['fine_amount'], 2),
            'total_amount' => round($totalAmount, 2),
            'fee_structures_applied' => $applicableFeeStructures->toArray(),
            'is_late_payment' => $fineCalculation['is_late_payment'],
            'due_date' => $fineCalculation['due_date'],
            'payment_date' => $paymentDate,
            'month' => $month,
            'year' => $year,
            'penalty_type' => $fineCalculation['penalty_type'],
            'penalty_percentage' => $fineCalculation['penalty_percentage']
        ];
    }

    /**
     * Busca estruturas de taxa aplicáveis para uma classe e mês
     */
    private function getApplicableFeeStructures(string $classroom, string $month): \Illuminate\Database\Eloquent\Collection
    {
        return FeeStructure::where('active', 1)
            ->get()
            ->filter(function ($feeStructure) use ($classroom, $month) {
                // Verifica se a classe está incluída
                $gradesArray = explode(',', $feeStructure->grades);
                $classIncluded = in_array($classroom, $gradesArray);
                
                // Verifica se o mês está incluído (se especificado)
                $monthIncluded = true;
                if (!empty($feeStructure->months)) {
                    $monthsArray = array_map('trim', explode(',', $feeStructure->months));
                    // Suportar comparação por número (1-12) ou nome (Janeiro-Dezembro)
                    $monthNumber = $this->getMonthNumber($month);
                    $monthIncluded = in_array((string) $monthNumber, $monthsArray)
                        || in_array($month, $monthsArray);
                }
                
                return $classIncluded && $monthIncluded;
            });
    }

    /**
     * Calcula multa baseada na estrutura de taxa e data de pagamento
     */
    private function calculateFine(
        \Illuminate\Database\Eloquent\Collection $feeStructures,
        float $baseFee,
        string $month,
        int $year,
        Carbon $paymentDate,
        ?PaymentReference $paymentReference = null
    ): array {
        $fineAmount = 0.0;
        $isLatePayment = false;
        $dueDate = null;
        $penaltyType = null;
        $penaltyPercentage = 0;

        foreach ($feeStructures as $feeStructure) {
            // Determina a data de vencimento
            $currentDueDate = $this->calculateDueDate($feeStructure, $month, $year, $paymentReference);
            
            if (!$dueDate || ($currentDueDate && $currentDueDate->lt($dueDate))) {
                $dueDate = $currentDueDate;
            }

            // Verifica se está em atraso
            if ($dueDate && $paymentDate->gt($dueDate)) {
                $isLatePayment = true;
                
                // Calcula multa baseada no tipo
                $currentPenaltyType = $feeStructure->penalty_type ?? 'percentage';
                $currentPenaltyValue = $feeStructure->late_penalty_percentage ?? 0;
                
                if ($currentPenaltyValue > 0) {
                    $penaltyType = $currentPenaltyType;
                    $penaltyPercentage = max($penaltyPercentage, $currentPenaltyValue);
                    
                    if (strtolower($currentPenaltyType) === 'fixed') {
                        $fineAmount += $currentPenaltyValue;
                    } else {
                        // Percentage (default)
                        $fineAmount += ($baseFee * $currentPenaltyValue) / 100;
                    }
                }
            }
        }

        return [
            'fine_amount' => $fineAmount,
            'is_late_payment' => $isLatePayment,
            'due_date' => $dueDate,
            'penalty_type' => $penaltyType,
            'penalty_percentage' => $penaltyPercentage
        ];
    }

    /**
     * Calcula data de vencimento baseada na estrutura de taxa
     */
    private function calculateDueDate(
        FeeStructure $feeStructure,
        string $month,
        int $year,
        ?PaymentReference $paymentReference = null
    ): ?Carbon {
        // Se PaymentReference tem expires_at, usa essa data
        if ($paymentReference && $paymentReference->expires_at) {
            try {
                return Carbon::parse($paymentReference->expires_at);
            } catch (\Exception $e) {
                Log::warning('Could not parse PaymentReference expires_at', [
                    'expires_at' => $paymentReference->expires_at,
                    'error' => $e->getMessage()
                ]);
            }
        }

        // Usa payment_due_day da FeeStructure
        $dueDay = $feeStructure->payment_due_day ?? 5; // Default: dia 5

        try {
            $monthNumber = $this->getMonthNumber($month);
            $dueDate = Carbon::createFromDate($year, $monthNumber, $dueDay)->endOfDay();

            // Mover para próximo dia útil se cair em fim de semana
            while ($dueDate->isWeekend()) {
                $dueDate->addDay();
            }

            return $dueDate;
        } catch (\Exception $e) {
            Log::warning('Could not calculate due date', [
                'month' => $month,
                'year' => $year,
                'due_day' => $dueDay,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Cria um pagamento preservando as multas no histórico
     */
    public function createFeePayment(
        User $student,
        array $paymentData,
        ?array $calculationData = null
    ): Fee_assign {
        // Se não foi fornecido cálculo, calcula agora
        if (!$calculationData) {
            $calculationData = $this->calculateFeeForStudent(
                $student,
                $paymentData['month'] ?? null,
                $paymentData['year'] ?? null,
                isset($paymentData['payment_date']) ? Carbon::parse($paymentData['payment_date']) : null
            );
        }

        $paidAmount = (float)($paymentData['amount'] ?? $calculationData['total_amount']);
        $manualFine = (float)($paymentData['fine'] ?? 0);
        $discount = (float)($paymentData['discount'] ?? 0);

        // A multa final é o maior valor entre a calculada e a manual
        $finalFine = max($calculationData['fine_amount'], $manualFine);

        // Distribui o valor pago
        $distribution = $this->distributePaidAmount(
            $calculationData['base_amount'],
            $finalFine,
            $paidAmount,
            $discount
        );

        $paymentDate = isset($paymentData['payment_date'])
            ? Carbon::parse($paymentData['payment_date'])
            : Carbon::now();

        $feeAssign = Fee_assign::create([
            'student_id' => $student->id,
            'fee_group_id' => $paymentData['fee_group_id'] ?? '1',

            // PRESERVA AS MULTAS NO HISTÓRICO
            'amount' => $distribution['base_allocated'],
            'fine' => $finalFine, // Multa preservada (não zerada)
            'discount' => $discount,

            // Informações do pagamento
            'payment_mode' => $paymentData['payment_mode'] ?? $paymentData['paymentMode'] ?? 'Cash',
            'pay_type' => $paymentData['pay_type'] ?? strtolower($paymentData['paymentMode'] ?? 'cash'),
            'status' => 'Paid',

            // Datas
            'payment_date' => $paymentDate,
            'day' => $paymentDate->day,
            'month' => $paymentData['month'] ?? $this->monthInPortuguese($paymentDate->month),
            'year' => $paymentData['year'] ?? $paymentDate->year,
            'pay_day' => $paymentDate->day,
            'pay_month' => $paymentDate->month,
            'pay_year' => $paymentDate->year,

            // Informações de transação
            'transaction_id' => $paymentData['transaction_id'] ?? null,
            'reference_number' => $paymentData['reference_number'] ?? null,
        ]);

        // Log removido para evitar excesso de entradas no laravel.log

        return $feeAssign;
    }

    /**
     * Distribui o valor pago entre taxa base e multa (mas preserva multa no histórico)
     */
    private function distributePaidAmount(
        float $baseAmount,
        float $fineAmount,
        float $paidAmount,
        float $discount = 0
    ): array {
        $expectedTotal = $baseAmount + $fineAmount - $discount;

        // SEMPRE salva apenas o valor base no campo amount
        // A multa é salva separadamente no campo fine
        // O total pago é amount + fine - discount

        if ($paidAmount >= $expectedTotal) {
            // Pagamento completo ou superior - salva apenas a base
            return [
                'base_allocated' => round($baseAmount, 2),
                'fine_preserved' => round($fineAmount, 2),
                'total_paid' => round($paidAmount, 2),
                'payment_type' => $paidAmount > $expectedTotal ? 'overpayment' : 'full_payment'
            ];
        } else {
            // Pagamento parcial
            $baseAllocated = max(0, min($baseAmount, $paidAmount - $discount));

            return [
                'base_allocated' => round($baseAllocated, 2),
                'fine_preserved' => round($fineAmount, 2), // Preserva multa mesmo em pagamento parcial
                'total_paid' => round($paidAmount, 2),
                'payment_type' => 'partial_payment'
            ];
        }
    }

    /**
     * Verifica se existe pagamento para um mês/ano
     */
    public function hasPaymentForPeriod(User $student, string $month, int $year): bool
    {
        return Fee_assign::where([
            'student_id' => $student->id,
            'month' => $month,
            'year' => $year,
            'status' => 'Paid'
        ])->exists();
    }

    /**
     * Busca dados de pagamento existente (compatibilidade com código antigo)
     * ATENÇÃO: Retorna apenas o primeiro pagamento. Use getPaymentsForPeriod() para todos
     */
    public function getPaymentForPeriod(User $student, string $month, int $year): ?Fee_assign
    {
        return Fee_assign::where([
            'student_id' => $student->id,
            'month' => $month,
            'year' => $year,
            'status' => 'Paid'
        ])->first();
    }

    /**
     * Busca TODOS os pagamentos de um período (novo método)
     * IMPORTANTE: Retorna array/collection de pagamentos separados por taxa
     */
    public function getPaymentsForPeriod(User $student, string $month, int $year)
    {
        return Fee_assign::where([
            'student_id' => $student->id,
            'month' => $month,
            'year' => $year,
            'status' => 'Paid'
        ])->get();
    }

    /**
     * Calcula resumo anual de taxas
     */
    public function calculateYearSummary(User $student, int $year): array
    {
        $months = [
            'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho',
            'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'
        ];

        $summary = [
            'total_expected' => 0,
            'total_paid' => 0,
            'total_pending' => 0,
            'total_overdue' => 0,
            'total_fines' => 0,
            'months_detail' => []
        ];

        foreach ($months as $month) {
            $calculation = $this->calculateFeeForStudent($student, $month, $year);
            $payment = $this->getPaymentForPeriod($student, $month, $year);

            // PULAR meses sem taxas atribuídas (base_amount = 0) E sem pagamento
            if ($calculation['base_amount'] == 0 && !$payment) {
                continue;
            }

            $isPaid = !is_null($payment);
            $paidAmount = $isPaid ? ($payment->amount + $payment->fine - $payment->discount) : 0;

            // Get fee names from fee structures
            $feeNames = collect($calculation['fee_structures_applied'] ?? [])
                ->pluck('name')
                ->join(', ');
            if (empty($feeNames)) {
                $feeNames = 'Taxa Escolar';
            }

            // Verifica se está em atraso APENAS se NÃO foi pago
            $isOverdue = false;
            $fineAmount = 0;

            if (!$isPaid && $calculation['due_date']) {
                $isOverdue = Carbon::now()->gt($calculation['due_date']);
                // Só aplica multa se estiver em atraso
                if ($isOverdue) {
                    $fineAmount = $calculation['fine_amount'];
                }
            } elseif ($isPaid && $payment) {
                // Se já foi pago, usa a multa registrada no pagamento
                $fineAmount = $payment->fine;
            }

            // Determine status
            $dataStatus = $isPaid ? 'paid' : ($isOverdue ? 'overdue' : 'pending');

            $monthData = [
                'month' => $month,
                'year' => $year,
                'base_amount' => $calculation['base_amount'],
                'fine_amount' => $fineAmount,
                'total_expected' => $calculation['base_amount'] + $fineAmount,
                'is_paid' => $isPaid,
                'paid_amount' => $paidAmount,
                'payment_data' => $payment,
                'payment_date' => $isPaid && $payment ? $payment->created_at : null,
                'is_overdue' => $isOverdue,
                'due_date' => $calculation['due_date'],
                'fee_names' => $feeNames,
                'discount_applied' => $isPaid && $payment ? $payment->discount : 0,
                'data_status' => $dataStatus,
                'existing_payment' => $payment
            ];

            $summary['total_expected'] += $calculation['base_amount'];
            $summary['total_fines'] += $fineAmount;

            if ($payment) {
                $summary['total_paid'] += $paidAmount;
            } else {
                if ($isOverdue) {
                    $summary['total_overdue'] += ($calculation['base_amount'] + $fineAmount);
                } else {
                    $summary['total_pending'] += $calculation['base_amount'];
                }
            }

            $summary['months_detail'][] = $monthData;
        }

        return $summary;
    }

    /**
     * Utilitários
     */
    private function defaultCalculation(float $base, float $fine): array
    {
        return [
            'base_amount' => $base,
            'fine_amount' => $fine,
            'total_amount' => $base + $fine,
            'fee_structures_applied' => [],
            'is_late_payment' => false,
            'due_date' => null,
            'payment_date' => Carbon::now(),
            'month' => $this->monthInPortuguese(Carbon::now()->month),
            'year' => Carbon::now()->year,
            'penalty_type' => null,
            'penalty_percentage' => 0
        ];
    }

    private function getMonthNumber(string $monthName): int
    {
        $months = [
            'january' => 1, 'february' => 2, 'march' => 3, 'april' => 4,
            'may' => 5, 'june' => 6, 'july' => 7, 'august' => 8,
            'september' => 9, 'october' => 10, 'november' => 11, 'december' => 12,
            'janeiro' => 1, 'fevereiro' => 2, 'março' => 3, 'abril' => 4,
            'maio' => 5, 'junho' => 6, 'julho' => 7, 'agosto' => 8,
            'setembro' => 9, 'outubro' => 10, 'novembro' => 11, 'dezembro' => 12
        ];

        $normalized = strtolower(trim($monthName));
        return $months[$normalized] ?? 1;
    }

    private function monthInPortuguese(int $monthNumber): string
    {
        $months = [
            1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
            5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
            9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
        ];

        return $months[$monthNumber] ?? 'Janeiro';
    }
}
