<?php

namespace App\Services;

use App\Models\IvaTransaction;
use Illuminate\Support\Facades\DB;

class IvaCalculatorService
{
    /**
     * Obter taxa de IVA das configurações (pos_settings)
     */
    public function getTaxRate(): float
    {
        $setting = DB::table('pos_settings')
            ->where('key', 'tax_rate')
            ->first();

        return $setting ? (float) $setting->value : 17.00;
    }

    /**
     * Verificar se IVA está habilitado no sistema
     */
    public function isIvaEnabled(): bool
    {
        $setting = DB::table('pos_settings')
            ->where('key', 'iva_enabled')
            ->first();

        return $setting ? (bool) $setting->value : false;
    }

    /**
     * Obter configuração de IVA para um tipo de fee (dinâmico)
     *
     * @param string|int $feeTypeIdentifier Nome, código ou ID do fee_type
     * @return array
     */
    public function getFeeTypeConfig($feeTypeIdentifier): array
    {
        // Buscar fee_type por nome, código ou ID
        $feeType = DB::table('fee_type')
            ->where(function($query) use ($feeTypeIdentifier) {
                if (is_numeric($feeTypeIdentifier)) {
                    $query->where('id', $feeTypeIdentifier);
                } else {
                    $query->where('name', 'LIKE', "%{$feeTypeIdentifier}%")
                          ->orWhere('fees_code', $feeTypeIdentifier);
                }
            })
            ->first();

        if (!$feeType) {
            // Padrão: não tem IVA (ex: propinas)
            return [
                'applicable' => false,
                'price_includes_iva' => false,
                'price' => 0,
                'rate' => $this->getTaxRate(),
                'fee_type_id' => null,
                'fee_type_name' => $feeTypeIdentifier,
            ];
        }

        return [
            'applicable' => (bool) $feeType->iva_applicable,
            'price_includes_iva' => (bool) $feeType->price_includes_iva,
            'price' => (float) ($feeType->default_price ?? 0),
            'rate' => $this->getTaxRate(),
            'fee_type_id' => $feeType->id,
            'fee_type_name' => $feeType->name,
        ];
    }
    /**
     * Calcula IVA baseado no preço e se IVA está incluído ou não
     *
     * @param float $price Preço mostrado
     * @param float $ivaRate Taxa de IVA (ex: 17.00 para 17%)
     * @param bool $priceIncludesIva Se o preço já inclui IVA
     * @return array ['base' => float, 'iva' => float, 'total' => float]
     */
    public function calculate(float $price, float $ivaRate, bool $priceIncludesIva): array
    {
        $ivaRate = $ivaRate / 100; // Converter 17 para 0.17

        if ($priceIncludesIva) {
            // Preço INCLUI IVA: preciso extrair o IVA
            // Total = Base * (1 + Taxa)
            // Base = Total / (1 + Taxa)
            $total = $price;
            $base = $total / (1 + $ivaRate);
            $iva = $total - $base;
        } else {
            // Preço NÃO INCLUI IVA: preciso adicionar o IVA
            // Base = Preço
            // IVA = Base * Taxa
            // Total = Base + IVA
            $base = $price;
            $iva = $base * $ivaRate;
            $total = $base + $iva;
        }

        return [
            'base' => round($base, 2),
            'iva' => round($iva, 2),
            'total' => round($total, 2),
        ];
    }

    /**
     * Exemplos de uso:
     *
     * 1. Certificado custa 100 MT SEM IVA (cliente paga 117 MT)
     *    calculate(100, 17, false) => ['base' => 100, 'iva' => 17, 'total' => 117]
     *
     * 2. Declaração custa 117 MT COM IVA incluído (base é 100 MT)
     *    calculate(117, 17, true) => ['base' => 100, 'iva' => 17, 'total' => 117]
     */

    /**
     * Sincronizar vendas POS para iva_transactions
     */
    public function syncPosSales(string $startDate, string $endDate): int
    {
        $sales = DB::table('pos_sales')
            ->whereBetween('sale_date', [$startDate, $endDate])
            ->where('status', 'completed')
            ->where('tax', '>', 0) // Apenas vendas com IVA
            ->get();

        $synced = 0;
        foreach ($sales as $sale) {
            // Verificar se já existe
            $exists = IvaTransaction::where('reference_number', $sale->sale_number)->exists();
            if ($exists) continue;

            $ivaAmount = (float) $sale->tax;
            $total = (float) $sale->total;
            $base = $total - $ivaAmount;

            IvaTransaction::create([
                'transaction_date' => date('Y-m-d', strtotime($sale->sale_date)),
                'transaction_type' => 'sale',
                'category' => 'pos',
                'description' => 'Venda POS #' . $sale->sale_number,
                'student_id' => $sale->user_id,
                'price_shown' => $total,
                'price_includes_iva' => true,
                'iva_rate' => $this->getTaxRate(),
                'base_amount' => $base,
                'iva_amount' => $ivaAmount,
                'total_amount' => $total,
                'reference_number' => $sale->sale_number,
                'related_id' => $sale->id,
                'related_type' => 'App\Models\PosSale',
                'created_by' => $sale->cashier_id,
            ]);
            $synced++;
        }

        return $synced;
    }

    /**
     * Sincronizar pagamentos de fees/propinas (incluindo certificados, uniformes, etc.)
     */
    public function syncFeePayments(string $startDate, string $endDate): int
    {
        $payments = DB::table('payment_references')
            ->whereBetween('paid_at', [$startDate, $endDate])
            ->where('status', 'paid')
            ->whereNotNull('paid_at')
            ->get();

        $synced = 0;
        foreach ($payments as $payment) {
            // Verificar se já existe
            $exists = IvaTransaction::where('reference_number', $payment->reference_number)->exists();
            if ($exists) continue;

            // Extrair tipo de fee do metadata
            $metadata = json_decode($payment->metadata, true);
            $feeTypeName = $metadata['custom_fee_type'] ?? $metadata['description'] ?? 'Propina';

            // Obter configuração de IVA para este tipo de fee (dinâmico)
            $config = $this->getFeeTypeConfig($feeTypeName);

            // Se não for sujeito a IVA, pular
            if (!$config['applicable']) {
                continue;
            }

            // Calcular IVA
            $amount = (float) $payment->amount + (float) ($payment->fine_amount ?? 0);
            $calculation = $this->calculate(
                $amount,
                $config['rate'],
                $config['price_includes_iva']
            );

            IvaTransaction::create([
                'transaction_date' => date('Y-m-d', strtotime($payment->paid_at)),
                'transaction_type' => 'sale',
                'category' => $feeTypeName,
                'description' => $feeTypeName . ' - ' . $payment->fee_month . '/' . $payment->fee_year,
                'student_id' => $payment->student_id,
                'price_shown' => $amount,
                'price_includes_iva' => $config['price_includes_iva'],
                'iva_rate' => $config['rate'],
                'base_amount' => $calculation['base'],
                'iva_amount' => $calculation['iva'],
                'total_amount' => $calculation['total'],
                'reference_number' => $payment->reference_number,
                'related_id' => $payment->id,
                'related_type' => 'App\Models\PaymentReference',
                'metadata' => json_encode($metadata),
            ]);
            $synced++;
        }

        return $synced;
    }

    /**
     * Sincronizar TODAS as transações do período
     */
    public function syncAllTransactions(string $startDate, string $endDate): array
    {
        return [
            'pos_sales' => $this->syncPosSales($startDate, $endDate),
            'fee_payments' => $this->syncFeePayments($startDate, $endDate),
        ];
    }

    /**
     * Gera relatório de IVA para um período (incluindo todas as fontes)
     *
     * @param string $startDate Data início
     * @param string $endDate Data fim
     * @param string|null $category Categoria específica ou null para todas
     * @param bool $autoSync Sincronizar automaticamente antes de gerar
     * @return array
     */
    public function generateReport(string $startDate, string $endDate, ?string $category = null, bool $autoSync = true): array
    {
        // Sincronizar transações antes de gerar relatório
        if ($autoSync) {
            $this->syncAllTransactions($startDate, $endDate);
        }

        $query = IvaTransaction::whereBetween('transaction_date', [$startDate, $endDate])
            ->where('transaction_type', 'sale');

        if ($category) {
            $query->where('category', $category);
        }

        $transactions = $query->get();

        $summary = [
            'period' => [
                'start' => $startDate,
                'end' => $endDate,
            ],
            'totals' => [
                'base_amount' => $transactions->sum('base_amount'),
                'iva_amount' => $transactions->sum('iva_amount'),
                'total_amount' => $transactions->sum('total_amount'),
                'transaction_count' => $transactions->count(),
            ],
            'by_category' => [],
            'by_rate' => [],
            'by_source' => [
                'pos' => 0,
                'certificado' => 0,
                'declaracao' => 0,
                'propina' => 0,
                'outros' => 0,
            ],
            'transactions' => $transactions,
        ];

        // Agrupar por categoria
        $byCategory = $transactions->groupBy('category');
        foreach ($byCategory as $cat => $items) {
            $summary['by_category'][$cat] = [
                'count' => $items->count(),
                'base' => $items->sum('base_amount'),
                'iva' => $items->sum('iva_amount'),
                'total' => $items->sum('total_amount'),
            ];

            // Também preencher by_source
            if (isset($summary['by_source'][$cat])) {
                $summary['by_source'][$cat] = $items->sum('iva_amount');
            } else {
                $summary['by_source']['outros'] += $items->sum('iva_amount');
            }
        }

        // Agrupar por taxa de IVA
        $byRate = $transactions->groupBy('iva_rate');
        foreach ($byRate as $rate => $items) {
            $summary['by_rate'][(float)$rate] = [
                'count' => $items->count(),
                'base' => $items->sum('base_amount'),
                'iva' => $items->sum('iva_amount'),
                'total' => $items->sum('total_amount'),
            ];
        }

        return $summary;
    }

    /**
     * Formata relatório para declaração oficial AT
     */
    public function formatForAT(array $reportData): array
    {
        // Formato específico para Autoridade Tributária de Moçambique
        return [
            'periodo' => $reportData['period']['start'] . ' a ' . $reportData['period']['end'],
            'iva_liquidado' => [
                'taxa_17' => $reportData['by_rate'][17.0] ?? ['base' => 0, 'iva' => 0],
                'taxa_5' => $reportData['by_rate'][5.0] ?? ['base' => 0, 'iva' => 0],
                'isento' => ['base' => 0, 'iva' => 0],
            ],
            'total_iva_liquidado' => $reportData['totals']['iva_amount'],
            'iva_dedutivel' => 0, // TODO: Implementar compras/despesas
            'iva_a_pagar' => $reportData['totals']['iva_amount'],
        ];
    }
}
