<?php

namespace App\Services;

use App\Models\PaymentReference;
use App\Models\User;
use App\Models\FailedReconciliation;
use App\Services\FeeCalculationService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class PaymentReconciliationService
{
    const HEADER_TYPE  = '0';
    const DETAIL_BMEPS = '1';
    const DETAIL_MEPS  = '2';
    const TRAILER_TYPE = '9';

    protected $feeCalculationService;

    public function __construct(FeeCalculationService $feeCalculationService)
    {
        $this->feeCalculationService = $feeCalculationService;
    }

    // ======= Diretórios =======
    private function baseDir(): string      { return public_path('bci/bmeps'); }
    private function inDir(): string        { return $this->baseDir() . '/incoming'; }
    private function processedDir(): string { return $this->baseDir() . '/processed'; }
    private function errorDir(): string     { return $this->baseDir() . '/failed'; }
    private function reportsDir(): string   { return $this->baseDir() . '/reports'; }

    // ======= Estado =======
    private array $processedReferences = [];
    private array $errors = [];
    private array $stats = [
        'total_files'           => 0,
        'total_records'         => 0,
        'reconciled'            => 0,
        'failed'                => 0,
        'duplicates'            => 0,
        'not_found'             => 0,
        'total_amount'          => 0.0,
        'autoscaled_10x'        => 0,
        'autoscaled_div10'      => 0,
        'with_calculated_fine'  => 0,
        'total_fines_preserved' => 0.0,
        'with_unified_service'  => 0,
    ];

    // ======= Entrada principal =======
    public function processAllPendingFiles(): array
    {
        Log::info('BMEPS reconciliation started');

        try {
            $files = $this->fetchLocalFiles();
            if (empty($files)) {
                return $this->stats;
            }

            Log::info('BMEPS files found', ['count' => count($files)]);

            foreach ($files as $file) {
                $this->processFile($file);
            }

            $this->generateReconciliationReport();
            Log::info('BMEPS reconciliation completed', ['reconciled' => $this->stats['reconciled'], 'failed' => $this->stats['failed']]);

        } catch (Exception $e) {
            Log::error('Global reconciliation failure', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
        }

        return $this->stats;
    }

    // ======= FS utils (mantidos iguais) =======
    private function ensureDirs(): void
    {
        foreach ([$this->inDir(), $this->processedDir(), $this->errorDir(), $this->reportsDir()] as $dir) {
            if (!is_dir($dir)) {
                @mkdir($dir, 0775, true);
            }
        }
    }

    private function fetchLocalFiles(): array
    {
        $this->ensureDirs();
        $files = glob($this->inDir() . '/*.txt', GLOB_NOSORT) ?: [];
        $files = array_merge($files, glob($this->inDir() . '/*.TXT', GLOB_NOSORT) ?: []);
        return array_unique($files);
    }

    private function moveFile(string $file, bool $success): void
    {
        // Sempre mover para processed/ - falhas são guardadas na BD (failed_reconciliations)
        // Isto evita acumulação de ficheiros na pasta failed/
        $destDir = $this->processedDir();
        $this->ensureDirs();

        $basename = basename($file);
        // Adiciona prefixo para indicar se processou com sucesso ou com erros
        $prefix = $success ? 'OK_' : 'ERR_';
        $target = $destDir . '/' . $prefix . now()->format('Ymd_His') . '_' . $basename;

        if (@rename($file, $target)) {
            // moved ok
        } elseif (@copy($file, $target)) {
            @unlink($file);
        } else {
            Log::warning('Could not move file', ['file' => basename($file)]);
        }
    }

    private function generateReconciliationReport(): string
    {
        $report = [
            'process_date' => now()->toDateTimeString(),
            'stats'        => $this->stats,
            'errors'       => $this->errors,
            'service_info' => [
                'unified_fee_calculation_service' => true,
                'preserves_fines_in_history' => true,
                'automatic_fine_calculation' => true
            ]
        ];

        $dir = $this->reportsDir();
        if (!file_exists($dir)) @mkdir($dir, 0775, true);

        $reportPath = $dir . '/rep_' . now()->format('Ymd_His') . '.json';
        @file_put_contents($reportPath, json_encode($report, JSON_PRETTY_PRINT));

        return $reportPath;
    }

    // ======= Core por ficheiro =======
    private function processFile(string $file): void
    {
        $this->stats['total_files']++;
        // Processing file

        try {
            $content = @file_get_contents($file);
            if ($content === false || $content === '') {
                throw new Exception("Empty or unreadable file");
            }

            $parsed = $this->parseFile($content);
            if (!$this->validateFileStructure($parsed)) {
                Log::warning('Skipping file with no detail records', ['file' => basename($file)]);
                $this->moveFile($file, false);
                return;
            }

            $this->saveFileInfo($file, $parsed);

            DB::beginTransaction();

            foreach ($parsed['details'] as $detail) {
                $this->processTransaction($detail, $file);
            }

            $this->validateTrailer($parsed);

            DB::commit();
            $this->finalizeLog(basename($file), 'completed');
            $this->moveFile($file, true);

        } catch (Exception $e) {
            DB::rollBack();

            Log::error('Failed to process file', ['file' => basename($file), 'error' => $e->getMessage()]);
            $this->errors[] = ['file' => basename($file), 'error' => $e->getMessage(), 'timestamp' => now()->toDateTimeString()];
            $this->finalizeLog(basename($file), 'failed');
            $this->moveFile($file, false);
        }
    }

    // ======= Negócio USANDO SERVICE UNIFICADO =======
    private function processTransaction(array $detail, string $file): void
    {
        if (empty($detail['reference'])) {
            Log::warning('Skipping detail without reference');
            return;
        }

        $this->stats['total_records']++;

        $reference = (string) $detail['reference'];
        $paidAmount = (float) $detail['amount'];

        // Apenas referências pendentes ou pagas (duplicados)
        // Referências expiradas NÃO são aceites — o aluno deve pagar a nova referência (com multa)
        $paymentReference = PaymentReference::where('reference_number', $reference)
            ->whereIn('status', ['pending', 'paid'])
            ->latest('updated_at')
            ->first();

        if (!$paymentReference) {
            $this->stats['not_found']++;
            Log::warning('Reference not found', ['reference' => $reference, 'file' => basename($file)]);
            $this->logUnmatchedPayment($detail);
            $this->logFailedReconciliation($detail, basename($file), 'Referência não encontrada no sistema');
            return;
        }

        if ($paymentReference->status === 'paid') {
            $this->stats['duplicates']++;
            Log::notice('Duplicate payment (already paid)', ['reference' => $reference]);
            return;
        }

        // Busca o estudante
        $student = User::find($paymentReference->student_id);
        if (!$student) {
            $this->stats['failed']++;
            Log::error('Student not found for payment reference', [
                'reference' => $reference,
                'student_id' => $paymentReference->student_id
            ]);
            $this->logFailedReconciliation($detail, basename($file), 'Estudante não encontrado (ID: ' . $paymentReference->student_id . ')');
            return;
        }

        // ✅ USA O SERVICE UNIFICADO para calcular multas
        $paymentDate = Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00'));
        
        try {
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $student,
                $paymentReference->fee_month,
                $paymentReference->fee_year,
                $paymentDate,
                $paymentReference
            );

            $this->stats['with_unified_service']++;

        } catch (\Exception $e) {
            // Fallback para valores da PaymentReference
            $calculation = [
                'base_amount' => $paymentReference->amount ?? $paidAmount,
                'fine_amount' => $paymentReference->fine_amount ?? 0,
                'total_amount' => ($paymentReference->amount ?? $paidAmount) + ($paymentReference->fine_amount ?? 0),
                'is_late_payment' => false
            ];
        }

        // Se FeeCalculationService retornou base_amount=0 mas a referência tem valor,
        // usar o valor da PaymentReference (ex: pagamento 'annual' sem FeeStructure)
        if (($calculation['base_amount'] ?? 0) == 0 && ($paymentReference->amount ?? 0) > 0) {
            $calculation['base_amount'] = (float) $paymentReference->amount;
            $calculation['fine_amount'] = (float) ($paymentReference->fine_amount ?? 0);
            $calculation['total_amount'] = $calculation['base_amount'] + $calculation['fine_amount'];
        }

        // Valida o montante pago com tolerância
        if (!$this->validatePaidAmount($calculation['total_amount'], $paidAmount)) {
            $this->stats['failed']++;
            $errorMessage = sprintf(
                'Valor pago (%.2f MT) não corresponde ao esperado (%.2f MT). Base: %.2f MT + Multa: %.2f MT',
                $paidAmount,
                $calculation['total_amount'],
                $calculation['base_amount'],
                $calculation['fine_amount']
            );
            Log::error('Payment amount validation failed', [
                'reference' => $reference,
                'paid_amount' => $paidAmount,
                'expected_total' => $calculation['total_amount'],
                'base_amount' => $calculation['base_amount'],
                'fine_amount' => $calculation['fine_amount']
            ]);
            $this->logFailedReconciliation($detail, basename($file), $errorMessage);
            return;
        }

        // Garantir transaction_id
        $this->ensureTransactionId($detail, basename($file));

        DB::transaction(function () use ($paymentReference, $detail, $paidAmount, $calculation, $paymentDate, $student) {
            // Atualiza a PaymentReference
            $paymentReference->status = 'paid';
            $paymentReference->paid_at = $paymentDate;
            $paymentReference->transaction_id = $detail['transaction_id'] ?? null;
            $paymentReference->terminal_id = $detail['terminal_id'] ?? null;
            
            // PRESERVA/atualiza valores de multa na PaymentReference para histórico
            if ($calculation['fine_amount'] > ($paymentReference->fine_amount ?? 0)) {
                $paymentReference->fine_amount = $calculation['fine_amount'];
                $this->stats['with_calculated_fine']++;
            }
            
            $paymentReference->save();

            // ✅ USA O SERVICE UNIFICADO para criar o pagamento preservando multas
            $paymentData = [
                'month' => $paymentReference->fee_month,
                'year' => $paymentReference->fee_year,
                'amount'=> (float) $calculation['total_amount'], // líquido (base + multa - desconto)
                'fine'  => (float) $calculation['fine_amount'],  // multa preservada
                'discount' => 0,
                'payment_mode' => 'Reference',
                'paymentMode' => 'Reference',
                'pay_type' => 'reference',
                'transaction_id' => $detail['transaction_id'] ?? null,
                'reference_number' => $detail['reference'] ?? null,
                'payment_date' => $paymentDate,
                'note' => 'Pagamento via reconciliação bancária (BMEPS/MEPS)'
            ];

            $feeAssign = $this->feeCalculationService->createFeePayment(
                $student,
                $paymentData,
                $calculation
            );
            
            $this->markAsProcessed($detail, $paidAmount);

            // Payment created ok
        });

        $this->stats['reconciled']++;
        $this->stats['total_amount'] += $paidAmount;
        $this->stats['total_fines_preserved'] += $calculation['fine_amount'];

        // Reconciled ok
    }

    // ======= Validação melhorada =======
    private function validatePaidAmount(float $expectedTotal, float $paidAmount): bool
    {
        $tolerance = (float) config('bmeps.amount_tolerance', 1.00);
        
        // Verifica se está dentro da tolerância
        if (abs($expectedTotal - $paidAmount) <= $tolerance) {
            return true;
        }

        // Tenta correção automática (x10 ou /10)
        $x10 = $paidAmount * 10;
        $div10 = $paidAmount / 10;

        if (abs($expectedTotal - $x10) <= $tolerance) {
            $this->stats['autoscaled_10x']++;
            return true;
        }

        if (abs($expectedTotal - $div10) <= $tolerance) {
            $this->stats['autoscaled_div10']++;
            return true;
        }

        return false;
    }

    // ======= Métodos auxiliares mantidos =======
    private function ensureTransactionId(array &$detail, string $fileName = null): void
    {
        if (!empty($detail['transaction_id'])) return;

        $date  = preg_replace('/\D+/', '', $detail['date'] ?? '');
        $time  = preg_replace('/\D+/', '', $detail['time'] ?? '');
        $stamp = ($date && $time) ? ($date.$time) : now()->format('YmdHis');
        $prefix= (isset($detail['type']) && $detail['type'] === '2') ? 'MEPS' : 'BMEPS';

        $detail['transaction_id'] = sprintf(
            '%s-%s-%s-%s',
            $prefix,
            $detail['reference'] ?? 'REF',
            $stamp,
            substr(sha1(($fileName ?? '').($detail['reference'] ?? '').$stamp), 0, 6)
        );
    }

    private function markAsProcessed(array $detail, float $amount): void
    {
        try {
            DB::table('payment_reconciliations')->insert([
                'transaction_id'    => $detail['transaction_id'] ?? null,
                'reference'         => $detail['reference'] ?? null,
                'amount'            => $amount,
                'payment_date'      => Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00')),
                'terminal_id'       => $detail['terminal_id'] ?? null,
                'terminal_location' => $detail['terminal_location'] ?? null,
                'processed_at'      => now(),
                'created_at'        => now(),
                'updated_at'        => now(),
            ]);
        } catch (\Throwable $e) {
            Log::error('payment_reconciliations insert failed', [
                'error'          => $e->getMessage(),
                'transaction_id' => $detail['transaction_id'] ?? null,
                'reference'      => $detail['reference'] ?? null,
                'amount'         => $amount,
            ]);
        }
    }

    private function logUnmatchedPayment(array $detail): void
    {
        try {
            $txnId = $detail['transaction_id'] ?? null;
            if (empty($txnId)) {
                $txnId = 'UNM_' . now()->format('YmdHis') . '_' . mt_rand(1000, 9999);
            }

            DB::table('unmatched_payments')->insert([
                'transaction_id'    => $txnId,
                'reference'         => $detail['reference'] ?? null,
                'amount'            => (float)($detail['amount'] ?? 0),
                'payment_date'      => Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00')),
                'terminal_id'       => $detail['terminal_id'] ?? null,
                'terminal_location' => $detail['terminal_location'] ?? null,
                'status'            => 'unmatched',
                'created_at'        => now()
            ]);
        } catch (\Throwable $e) {
            Log::warning('Could not write into unmatched_payments (optional table).', ['error' => $e->getMessage()]);
        }
    }

    private function logFailedReconciliation(array $detail, string $fileName, string $errorReason): void
    {
        try {
            FailedReconciliation::create([
                'reference_number' => $detail['reference'] ?? null,
                'transaction_id' => $detail['transaction_id'] ?? null,
                'terminal_id' => $detail['terminal_id'] ?? null,
                'amount' => (float)($detail['amount'] ?? 0),
                'payment_date' => Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00'))->format('Y-m-d'),
                'entity_code' => '90013',
                'error_reason' => $errorReason,
                'file_name' => $fileName,
                'raw_content' => $detail['raw'] ?? null,
                'status' => 'pending',
            ]);

            // Logged to failed_reconciliations
        } catch (\Throwable $e) {
            Log::error('Could not log failed reconciliation', [
                'error' => $e->getMessage(),
                'reference' => $detail['reference'] ?? null
            ]);
        }
    }

    private function saveFileInfo(string $filePath, array $parsedData): void
    {
        $totalAmount = array_reduce($parsedData['details'], fn($a, $d) => $a + (float)$d['amount'], 0.0);

        $fileDate = $parsedData['header']['creation_date'] ?? (
            !empty($parsedData['details']) ? min(array_map(fn($d) => $d['date'] ?? date('Y-m-d'), $parsedData['details'])) : date('Y-m-d')
        );

        try {
            DB::table('reconciliation_logs')->insert([
                'file_name'     => basename($filePath),
                'file_id'       => $parsedData['header']['file_id'] ?? null,
                'file_date'     => $fileDate,
                'total_records' => count($parsedData['details']),
                'total_amount'  => $totalAmount,
                'status'        => 'processing',
                'started_at'    => now(),
                'created_at'    => now()
            ]);
        } catch (\Throwable $e) {
            Log::warning('Could not write into reconciliation_logs (optional table).', ['error' => $e->getMessage()]);
        }
    }

    private function finalizeLog(string $fileName, string $status = 'completed'): void
    {
        try {
            DB::table('reconciliation_logs')
                ->where('file_name', $fileName)
                ->orderByDesc('started_at')
                ->limit(1)
                ->update([
                    'reconciled'   => $this->stats['reconciled'],
                    'failed'       => $this->stats['failed'],
                    'duplicates'   => $this->stats['duplicates'],
                    'not_found'    => $this->stats['not_found'],
                    'total_amount' => $this->stats['total_amount'],
                    'status'       => $status,
                    'completed_at' => now(),
                    'updated_at'   => now(),
                ]);
        } catch (\Throwable $e) {
            Log::warning('Could not finalize reconciliation_logs row.', ['file' => $fileName, 'error' => $e->getMessage()]);
        }
    }

    // ======= Parsers mantidos iguais =======
    private function parseFile(string $content): array
    {
        $lines  = preg_split('/\r\n|\r|\n/', rtrim($content));
        $result = ['header' => null, 'details' => [], 'trailer' => null];

        // Detecta MEPS (header contém "MEPS" OU existe linha tipo '2')
        $isMEPS = false;
        if (!empty($lines)) {
            $first = $lines[0];
            if (stripos($first, 'MEPS') !== false) {
                $isMEPS = true;
            } else {
                foreach ($lines as $l) {
                    if (isset($l[0]) && $l[0] === self::DETAIL_MEPS) { $isMEPS = true; break; }
                }
            }
        }
        // Format detected: MEPS or BMEPS

        foreach ($lines as $i => $raw) {
            $line = rtrim($raw);
            if ($line === '') continue;

            $type = substr($line, 0, 1);

            if ($type === self::HEADER_TYPE) {
                $result['header'] = $isMEPS ? $this->parseHeaderMEPS($line) : $this->parseHeaderBMEPS($line);
                continue;
            }
            if ($type === self::TRAILER_TYPE) {
                $result['trailer'] = $isMEPS ? $this->parseTrailerMEPS($line) : $this->parseTrailerBMEPS($line);
                continue;
            }

            if ($isMEPS) {
                if ($type === self::DETAIL_MEPS) {
                    $d = $this->parseDetailMEPS($line);
                    if ($d) $result['details'][] = $d;
                }
            } else {
                if ($type === self::DETAIL_BMEPS) {
                    $d = $this->parseDetailBMEPS($line);
                    if ($d) $result['details'][] = $d;
                }
            }
        }

        // Se header sem data, usa a menor data dos detalhes
        if (($result['header']['creation_date'] ?? null) === null && !empty($result['details'])) {
            $minDate = min(array_map(fn($d) => $d['date'] ?? date('Y-m-d'), $result['details']));
            $result['header']['creation_date'] = $minDate;
        }

        // Auto-correção pelo trailer quando há 1 único detalhe
        if ($result['trailer'] && count($result['details']) === 1) {
            $detSum = (float) $result['details'][0]['amount'];
            $trlSum = (float) $result['trailer']['total_amount'];

            if (abs($detSum - $trlSum) > 0.009) {
                if (abs(($detSum * 10) - $trlSum) <= 0.009) {
                    $result['details'][0]['amount'] = $detSum * 10;
                }
            }
        }

        return $result;
    }

    // ======= Parsers específicos (mantidos iguais) =======
    private function parseHeaderBMEPS(string $line): array
    {
        return [
            'type'          => '0',
            'format'        => 'BMEPS',
            'company_id'    => substr($line, 1, 3),
            'profile_id'    => substr($line, 4, 2),
            'creation_date' => $this->parseDateToYmd(substr($line, 6, 8)),
            'file_id'       => substr($line, 14, 5),
            'raw'           => $line,
        ];
    }

    private function parseDetailBMEPS(string $line): ?array
    {
        $reference  = $this->normalizeReference(substr($line, 1, 11) ?: substr($line, 72, 11));
        $amount     = $this->parseAmount(substr($line, 12, 16));
        $commission = $this->parseAmount(substr($line, 28, 16));
        $date       = $this->parseDateToYmd(substr($line, 44, 8));
        $time       = $this->parseTime(substr($line, 52, 6));
        $txnId      = trim(substr($line, 58, 7));
        $termId     = trim(substr($line, 65, 10));
        $location   = rtrim(substr($line, 75, 16) ?: '');

        if (!$reference) return null;

        return [
            'type'              => '1',
            'reference'         => $reference,
            'amount'            => $amount,
            'commission'        => $commission,
            'date'              => $date,
            'time'              => $time,
            'transaction_id'    => $txnId ?: null,
            'terminal_id'       => $termId ?: null,
            'terminal_location' => $location ?: null,
            'raw'               => $line,
        ];
    }

    private function parseTrailerBMEPS(string $line): array
    {
        return [
            'type'             => '9',
            'total_records'    => (int) substr($line, 1, 7),
            'total_amount'     => $this->parseAmount(substr($line, 8, 16)),
            'total_commission' => $this->parseAmount(substr($line, 24, 16)),
            'raw'              => $line,
        ];
    }

    private function parseHeaderMEPS(string $line): array
    {
        $date = null;
        if (preg_match('/(20\d{6})/', $line, $m)) {
            $date = $this->parseDateToYmd($m[1]);
        }

        return [
            'type'          => '0',
            'format'        => 'MEPS',
            'creation_date' => $date,
            'raw'           => $line,
        ];
    }

            private function parseDetailMEPS(string $line): ?array
            {
                $re = '/^2.*?'
                    . '(?P<date>20\d{6})'
                    . '(?P<time>\d{4}|\d{6})'
                    . '(?P<amount>\d{12})'
                    . '(?P<commission>\d{10,12})'
                    . '\s{0,3}'
                    . '0{10,}'
                    . '\s*'
                    . '(?P<ref>\d{11})'
                    . '00\s*'  // Fixed: added missing concatenation operator
                    . '/';

                $trimmed = rtrim($line);
                if (!preg_match($re, $trimmed, $m)) {
                    return null;
                }

                $reference  = $this->normalizeReference($m['ref']);
                $amount     = ((int)$m['amount']) / 100.0;  // MEPS: valor em centavos, dividir por 100
                $commission = ((int)$m['commission']) / 100.0;
                $date       = $this->parseDateToYmd($m['date']);

                $timeRaw = $m['time'];
                $time = (strlen($timeRaw) === 4)
                    ? substr($timeRaw,0,2).':'.substr($timeRaw,2,2).':00'
                    : $this->parseTime($timeRaw);

                if (!$reference) return null;

                return [
                    'type'              => '2',
                    'reference'         => $reference,
                    'amount'            => $amount,
                    'commission'        => $commission,
                    'date'              => $date,
                    'time'              => $time,
                    'transaction_id'    => null,
                    'terminal_id'       => null,
                    'terminal_location' => null,
                    'raw'               => $line,
                ];
            }

    private function parseTrailerMEPS(string $line): array
    {
        $count     = (int) substr($line, 1, 8);
        $totalAmt  = (int) substr($line, 9, 16)  / 100.0;
        $totalComm = (int) substr($line, 25, 16) / 100.0;

        return [
            'type'             => '9',
            'total_records'    => $count,
            'total_amount'     => $totalAmt,
            'total_commission' => $totalComm,
            'raw'              => $line,
        ];
    }

    // ======= Helpers =======
    private function parseDateToYmd(string $dateStr): string
    {
        $s = preg_replace('/\D+/', '', $dateStr);
        if (strlen($s) !== 8) return date('Y-m-d');

        $yyyy = (int)substr($s, 0, 4);
        $mm   = (int)substr($s, 4, 2);
        $dd   = (int)substr($s, 6, 2);

        if ($yyyy >= 1900 && $yyyy <= 2100) {
            return sprintf('%04d-%02d-%02d', $yyyy, $mm, $dd);
        }

        $d = (int)substr($s, 0, 2);
        $m = (int)substr($s, 2, 2);
        $y = (int)substr($s, 4, 4);
        return sprintf('%04d-%02d-%02d', $y, $m, $d);
    }

    private function parseTime(string $timeStr): string
    {
        $s = preg_replace('/\D+/', '', $timeStr);
        if (strlen($s) !== 6) return '00:00:00';
        $h = (int)substr($s, 0, 2);
        $i = (int)substr($s, 2, 2);
        $u = (int)substr($s, 4, 2);
        return sprintf('%02d:%02d:%02d', $h, $i, $u);
    }

    private function parseAmount(string $amountStr): float
    {
        $n = (int)preg_replace('/\D+/', '', $amountStr);
        return $n / 100.0;
    }

    /**
     * Normaliza referência para formato padrão (V1 - 11 dígitos)
     * MANTIDO para compatibilidade com código existente
     *
     * @deprecated Use normalizeReferenceV2() para suportar 11 e 13 dígitos
     */
    private function normalizeReference(?string $ref): ?string
    {
        if ($ref === null) return null;

        $ref = preg_replace('/\D+/', '', $ref);
        if (substr($ref, -2) === '00') {
            $ref = substr($ref, 0, -2);
        }
        if (strlen($ref) > 11) {
            $ref = substr($ref, -11);
        }
        if (strlen($ref) < 11) {
            $ref = str_pad($ref, 11, '0', STR_PAD_LEFT);
        }

        return $ref;
    }

    /**
     * Normaliza referência V2 - aceita 11 OU 13 dígitos
     * Recomendado para uso com novo formato de referências
     *
     * @param string|null $ref Referência a normalizar
     * @return string|null Referência normalizada (11 ou 13 dígitos)
     */
    private function normalizeReferenceV2(?string $ref): ?string
    {
        if ($ref === null) return null;

        // Remove caracteres não numéricos
        $ref = preg_replace('/\D+/', '', $ref);

        // Remove sufixo '00' se presente
        if (substr($ref, -2) === '00') {
            $ref = substr($ref, 0, -2);
        }

        $length = strlen($ref);

        if ($length === 11 || $length === 13) {
            return $ref;
        }

        if ($length > 13) {
            return substr($ref, -13);
        }

        if ($length > 11 && $length < 13) {
            return substr($ref, -11);
        }

        if ($length < 11) {
            return str_pad($ref, 11, '0', STR_PAD_LEFT);
        }

        return $ref;
    }

    /**
     * Teste público para normalizeReference (V1)
     * @param string $ref Referência a normalizar
     * @return string Referência normalizada
     */
    public function testNormalizeReferenceV1(string $ref): string
    {
        return $this->normalizeReference($ref);
    }

    /**
     * Teste público para normalizeReferenceV2
     * @param string $ref Referência a normalizar
     * @return string Referência normalizada
     */
    public function testNormalizeReferenceV2(string $ref): string
    {
        return $this->normalizeReferenceV2($ref);
    }

    /**
     * Teste de normalização V1 vs V2
     *
     * @return array Resultados dos testes
     */
    public function testNormalizationV1vsV2(): array
    {
        $testCases = [
            '1234567012547' => '13 digits (V2 format with year)',
            '12345670102' => '11 digits (V1 format no year)',
            '123456701254700' => '15 digits (will be truncated)',
            '123456701' => '9 digits (will be padded)',
        ];

        $results = [];

        foreach ($testCases as $input => $description) {
            $v1Result = $this->normalizeReference($input);
            $v2Result = $this->normalizeReferenceV2($input);

            $results[] = [
                'input' => $input,
                'description' => $description,
                'input_length' => strlen($input),
                'v1_normalized' => $v1Result,
                'v1_length' => strlen($v1Result),
                'v2_normalized' => $v2Result,
                'v2_length' => strlen($v2Result),
                'different' => $v1Result !== $v2Result,
                'v2_preserves_13_digits' => (strlen($input) === 13 && strlen($v2Result) === 13)
            ];
        }

        return [
            'test_cases' => $results,
            'summary' => [
                'v1_always_11_digits' => true,
                'v2_accepts_both_11_and_13' => true,
                'recommendation' => 'Use normalizeReferenceV2() para suportar ambos formatos'
            ]
        ];
    }

    private function validateFileStructure(array $parsedData): bool
    {
        if (count($parsedData['details']) === 0) {
            return false;
        }
        return true;
    }

    private function validateTrailer(array $parsedData): bool
    {
        if (!$parsedData['trailer']) {
            return true;
        }

        $trailer = $parsedData['trailer'];
        $details = $parsedData['details'];

        $expected = count($details);
        if ((int)$trailer['total_records'] !== $expected) {
            // Record count mismatch - tracked in stats
        }

        $sum = array_reduce($details, fn($a,$d)=>$a + (float)$d['amount'], 0.0);
        if (abs($sum - (float)$trailer['total_amount']) > 0.01) {
            // Amount total mismatch - tracked in stats
        }

        return true;
    }
}
