/**
 * SIDEBAR MODERNO - JavaScript
 * Versão: 2.0.0
 * Funcionalidades: Toggle mobile, accordion menu, active states
 */

(function() {
    'use strict';

    // Esperar DOM carregar
    document.addEventListener('DOMContentLoaded', function() {

        // ===== CRIAR BOTÃO TOGGLE MOBILE =====
        const toggleButton = document.createElement('button');
        toggleButton.className = 'sidebar-toggle';
        toggleButton.innerHTML = '<i class="fas fa-bars"></i>';
        toggleButton.setAttribute('aria-label', 'Toggle Menu');
        document.body.appendChild(toggleButton);

        const sidebar = document.querySelector('.page-sidebar');
        const body = document.body;

        // ===== TOGGLE SIDEBAR MOBILE =====
        toggleButton.addEventListener('click', function() {
            sidebar.classList.toggle('active');
            body.classList.toggle('sidebar-open');

            // Mudar ícone
            const icon = this.querySelector('i');
            if (sidebar.classList.contains('active')) {
                icon.className = 'fas fa-times';
            } else {
                icon.className = 'fas fa-bars';
            }
        });

        // Fechar ao clicar no overlay
        body.addEventListener('click', function(e) {
            if (body.classList.contains('sidebar-open') &&
                !sidebar.contains(e.target) &&
                !toggleButton.contains(e.target)) {
                sidebar.classList.remove('active');
                body.classList.remove('sidebar-open');
                toggleButton.querySelector('i').className = 'fas fa-bars';
            }
        });

        // ===== MENU ACCORDION =====
        const menuItems = document.querySelectorAll('.accordion-menu > li > a');

        menuItems.forEach(function(item) {
            item.addEventListener('click', function(e) {
                const parent = this.parentElement;
                const submenu = parent.querySelector('ul');

                // Se tem submenu
                if (submenu) {
                    e.preventDefault();

                    // Fechar outros menus
                    const allMenus = document.querySelectorAll('.accordion-menu > li');
                    allMenus.forEach(function(menu) {
                        if (menu !== parent && menu.classList.contains('open')) {
                            menu.classList.remove('open');
                        }
                    });

                    // Toggle este menu
                    parent.classList.toggle('open');
                } else {
                    // Fechar sidebar em mobile após clicar
                    if (window.innerWidth <= 1024) {
                        sidebar.classList.remove('active');
                        body.classList.remove('sidebar-open');
                        toggleButton.querySelector('i').className = 'fas fa-bars';
                    }
                }
            });
        });

        // ===== MARCAR ITEM ATIVO =====
        const currentUrl = window.location.href;
        const allLinks = document.querySelectorAll('.accordion-menu a');

        allLinks.forEach(function(link) {
            if (link.href === currentUrl) {
                link.classList.add('active');

                // Abrir submenu pai se existir
                const parentLi = link.closest('.accordion-menu > li');
                if (parentLi) {
                    parentLi.classList.add('open');

                    // Marcar o menu pai também
                    const parentLink = parentLi.querySelector('> a');
                    if (parentLink) {
                        parentLink.classList.add('active-parent');
                    }
                }
            }
        });

        // ===== FECHAR SUBMENU AO REDIMENSIONAR =====
        let resizeTimer;
        window.addEventListener('resize', function() {
            clearTimeout(resizeTimer);
            resizeTimer = setTimeout(function() {
                if (window.innerWidth > 1024) {
                    sidebar.classList.remove('active');
                    body.classList.remove('sidebar-open');
                    toggleButton.querySelector('i').className = 'fas fa-bars';
                }
            }, 250);
        });

        // ===== SMOOTH SCROLL PARA LINKS ÂNCORA =====
        document.querySelectorAll('.accordion-menu a[href^="#"]').forEach(function(anchor) {
            anchor.addEventListener('click', function(e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });

        // ===== DETECTAR SCROLL NO SIDEBAR =====
        sidebar.addEventListener('scroll', function() {
            if (this.scrollTop > 50) {
                this.classList.add('scrolled');
            } else {
                this.classList.remove('scrolled');
            }
        });

        // ===== PREVENIR SCROLL DO BODY QUANDO SIDEBAR ABERTO (MOBILE) =====
        let scrollPosition = 0;

        function lockScroll() {
            scrollPosition = window.pageYOffset;
            body.style.overflow = 'hidden';
            body.style.position = 'fixed';
            body.style.top = `-${scrollPosition}px`;
            body.style.width = '100%';
        }

        function unlockScroll() {
            body.style.removeProperty('overflow');
            body.style.removeProperty('position');
            body.style.removeProperty('top');
            body.style.removeProperty('width');
            window.scrollTo(0, scrollPosition);
        }

        // Aplicar lock/unlock conforme necessário
        const observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.attributeName === 'class') {
                    if (body.classList.contains('sidebar-open') && window.innerWidth <= 1024) {
                        lockScroll();
                    } else {
                        unlockScroll();
                    }
                }
            });
        });

        observer.observe(body, {
            attributes: true
        });

        // ===== ADICIONAR BADGES (OPCIONAL) =====
        // Exemplo de como adicionar badges dinamicamente
        function addBadge(linkSelector, count) {
            const link = document.querySelector(linkSelector);
            if (link && count > 0) {
                const badge = document.createElement('span');
                badge.className = 'sidebar-badge';
                badge.textContent = count;
                link.appendChild(badge);
            }
        }

        // Exemplos de uso (descomentar e ajustar conforme necessário):
        // addBadge('a[href*="notifications"]', 5);
        // addBadge('a[href*="messages"]', 12);

        // ===== HIGHLIGHT AO PASSAR MOUSE =====
        const menuLinks = document.querySelectorAll('.accordion-menu a');
        menuLinks.forEach(function(link) {
            link.addEventListener('mouseenter', function() {
                this.style.setProperty('--hover-x', '0%');
            });
        });

        console.log('✅ Sidebar Moderno carregado com sucesso!');
    });

    // ===== HELPERS =====

    // Debounce function
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    // Check if mobile
    function isMobile() {
        return window.innerWidth <= 1024;
    }

    // Get active menu item
    function getActiveMenuItem() {
        return document.querySelector('.accordion-menu a.active');
    }

})();
