<?php

namespace App\Http\Livewire\Admin\Account;

use App\Models\Admin;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;

class AdminPassword extends Component
{
    use FindGuard, LivewireAlert;

    public $user;
    public $password_confirmation;
    public $password;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function mount()
    {
        try {
            $this->user = $this->findGuardType()->user();
        } catch (Exception $e) {
            $this->alert('error', 'Try again an error occurred.');
        }
    }

    protected function rules(): array
    {
        return [
            'password' => ['required', 'confirmed', 'same:password_confirmation', Password::min(8)
                ->mixedCase()
                ->letters()
                ->numbers()
                ->symbols()
            ]
        ];
    }

    /**
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    /**
     * update user password here
     */
    public function submit()
    {
        $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        // Check if current password is same as the new one
        if (Hash::check($this->password, $this->user->getAuthPassword())) {
            $this->reset(['password', 'password_confirmation']);
            $this->alert('warning', 'Desculpe, a nova senha é igual à atual.');
            return redirect()->back();
        }

        $this->user->update([
            'password' => bcrypt($this->password)
        ]);

        Note::createSystemNotification(Admin::class, 'Atualização de senha', 'Credenciais atualizadas com sucesso.');
        $this->emit('noteAdded');
        $this->reset(['password', 'password_confirmation']);
        $this->alert('success', 'Credenciais atualizadas com sucesso.');
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function render()
    {
        return view('livewire.admin.account.admin-password');
    }
}
