<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use App\Services\SmsService;
use Carbon\Carbon;

class SendBirthdayMessages extends Command
{
    protected $signature = 'send:birthday-messages';
    protected $description = 'Envia mensagens de feliz aniversário aos estudantes';

    /**
     * Caminho do ficheiro de log dedicado
     */
    private string $logFile;

    public function __construct()
    {
        parent::__construct();
        $this->logFile = storage_path('logs/birthdays.log');
    }

    public function handle()
    {
        $this->log('=== Mensagens de Aniversário ===');
        $this->log('Data/Hora: ' . now()->format('d/m/Y H:i:s'));

        $today = Carbon::today()->format('m-d'); // Obtém MM-DD
        $this->log("Procurando aniversariantes do dia: {$today}");

        $students = User::where('is_active', 1)
            ->whereRaw("DATE_FORMAT(dob, '%m-%d') = ?", [$today])
            ->get();

        if ($students->isEmpty()) {
            $this->log('Nenhum estudante faz aniversário hoje.');
            $this->info('Nenhum estudante faz aniversário hoje.');
            return 0;
        }

        $this->log("Encontrados {$students->count()} aniversariante(s)");
        $this->info("Encontrados {$students->count()} aniversariante(s)");

        $smsService = new SmsService();
        $this->log("Gateway SMS: {$smsService->getGateway()}");

        $sent = 0;
        $failed = 0;

        foreach ($students as $student) {
            if (!$student->phone_number) {
                $this->log("   ✗ Sem telefone: {$student->name}");
                $failed++;
                continue;
            }

            $message = "Feliz Aniversario, {$student->name}! Que sua trajetoria academica seja repleta de conquistas e sucesso. Votos do COPMOZ";

            try {
                $result = $smsService->send($student->phone_number, $message);

                if ($result['success']) {
                    $this->log("   ✓ Enviado: {$student->name} ({$student->phone_number})");
                    $this->info("✓ {$student->name}");
                    $sent++;
                } else {
                    $this->log("   ✗ Erro para {$student->name}: " . ($result['error'] ?? 'Erro desconhecido'));
                    $this->error("✗ {$student->name}: " . ($result['error'] ?? 'Erro desconhecido'));
                    $failed++;
                }

            } catch (\Exception $e) {
                $this->log("   ✗ Erro para {$student->name}: {$e->getMessage()}");
                $this->error("✗ {$student->name}: {$e->getMessage()}");
                $failed++;

                Log::error("Birthday message failed", [
                    'student_id' => $student->id,
                    'student_name' => $student->name,
                    'error' => $e->getMessage()
                ]);
            }
        }

        $this->log('--- Resumo ---');
        $this->log("Enviados: {$sent}");
        $this->log("Falhas: {$failed}");
        $this->log('');

        $this->newLine();
        $this->info("Resumo: {$sent} enviados, {$failed} falhas");

        return 0;
    }

    /**
     * Escreve no ficheiro de log dedicado
     */
    private function log(string $message): void
    {
        $timestamp = now()->format('Y-m-d H:i:s');
        $logMessage = "[{$timestamp}] {$message}" . PHP_EOL;

        file_put_contents($this->logFile, $logMessage, FILE_APPEND);
    }
}

