<?php

namespace App\Http\Controllers;

use App\Models\Attendance;
use App\Models\Classroom;
use App\Models\Downloadsetings;
use App\Models\Exam;
use App\Models\Guardian;
use App\Models\GuardianDeviceToken;
use App\Models\Noticeboard;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class GuardianApiController extends Controller
{
    /**
     * Login do encarregado
     * POST /api/guardian/login
     * Body: { phone, password }
     */
    public function login(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'phone'    => 'required',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $phone = preg_replace('/\D/', '', $request->phone);
        if (strlen($phone) === 9) {
            $phone = '258' . $phone;
        }

        $guardian = Guardian::where('phone', $phone)->first();

        if (!$guardian || !Hash::check($request->password, $guardian->password)) {
            return response()->json([
                'status' => false,
                'error'  => 'Credenciais invalidas. Verifique o telefone e a senha.',
            ], 401);
        }

        if (!$guardian->is_active) {
            return response()->json([
                'status' => false,
                'error'  => 'Conta inactiva. Contacte a secretaria.',
            ], 403);
        }

        $guardian->update(['last_login_at' => now()]);

        $token = $guardian->createToken('Guardian API Token')->accessToken;

        return response()->json([
            'status' => true,
            'token'  => $token,
            'guardian' => [
                'id'           => $guardian->id,
                'name'         => $guardian->name,
                'phone'        => $guardian->phone,
                'email'        => $guardian->email,
                'relationship' => $guardian->relationship,
            ],
        ], 200);
    }

    /**
     * Retorna o guardian autenticado ou 401
     */
    private function guardianOrFail(): ?Guardian
    {
        return Auth::guard('guardian-api')->user();
    }

    /**
     * Listar educandos do encarregado
     * GET /api/guardian/students
     */
    public function listStudents(Request $request): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        $students = $guardian->students()->with('classroom')->get();

        return response()->json([
            'status'   => true,
            'students' => $students->map(function ($st) {
                return [
                    'id'         => $st->id,
                    'name'       => $st->name,
                    'student_id' => $st->student_id,
                    'sex'        => $st->sex,
                    'avatar'     => $st->avatar,
                    'class'      => $st->classroom?->class,
                    'classroom'  => $st->classroom?->name,
                    'is_active'  => $st->is_active,
                    'relationship' => $st->pivot->relationship,
                    'is_primary'   => $st->pivot->is_primary,
                ];
            }),
        ], 200);
    }

    /**
     * Resumo de um estudante
     * GET /api/guardian/student/{id}
     */
    public function studentDetail(Request $request, string $id): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        if (!$guardian->hasStudent($id)) {
            return response()->json(['status' => false, 'message' => 'Acesso negado'], 403);
        }

        $student = User::with('classroom')->find($id);
        if (!$student) {
            return response()->json(['status' => false, 'message' => 'Estudante nao encontrado'], 404);
        }

        return response()->json([
            'status'  => true,
            'student' => [
                'id'         => $student->id,
                'name'       => $student->name,
                'student_id' => $student->student_id,
                'sex'        => $student->sex,
                'dob'        => $student->dob,
                'avatar'     => $student->avatar,
                'class'      => $student->classroom?->class,
                'classroom'  => $student->classroom?->name,
                'father_name'  => $student->father_name,
                'mother_name'  => $student->mother_name,
                'is_active'    => $student->is_active,
            ],
        ], 200);
    }

    /**
     * Notas de um estudante
     * GET /api/guardian/student/{id}/notas?year=2026
     */
    public function studentExams(Request $request, string $id): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        if (!$guardian->hasStudent($id)) {
            return response()->json(['status' => false, 'message' => 'Acesso negado'], 403);
        }

        $year = $request->query('year', now()->year);

        $exams = Exam::with('subject')
            ->where('student_id', $id)
            ->where(function ($q) use ($year) {
                $q->where('year', $year)
                  ->orWhere('created_at', 'LIKE', '%' . $year . '%');
            })
            ->get();

        if ($exams->isEmpty()) {
            return response()->json([
                'status' => true,
                'exams'  => [],
                'message' => 'Sem notas registadas.',
            ], 200);
        }

        $grouped = $exams->groupBy(['trimester_id', 'subject_id']);
        $responseData = [];

        foreach ($grouped as $trimesterId => $subjects) {
            $trimesterData = [];
            foreach ($subjects as $subjectId => $examGroup) {
                $subjectData = [
                    'subject_id'   => $subjectId,
                    'subject_name' => $examGroup->first()->subject->name ?? 'N/A',
                    'exams'        => [],
                ];

                foreach ($examGroup as $exam) {
                    $subjectData['exams'][] = [
                        'id'   => $exam->id,
                        'ACS1' => $exam->ACS1a,
                        'ACS2' => $exam->ACS2a,
                        'MACS' => $exam->MACS,
                        'AT'   => $exam->AT,
                        'MT'   => $exam->MT,
                        'NE'   => $exam->NE,
                        'year' => $exam->year,
                    ];
                }

                $trimesterData[] = $subjectData;
            }

            $responseData[] = [
                'trimester_id' => $trimesterId,
                'subjects'     => $trimesterData,
            ];
        }

        return response()->json([
            'status' => true,
            'year'   => $year,
            'exams'  => $responseData,
        ], 200);
    }

    /**
     * Pagamentos de um estudante
     * GET /api/guardian/student/{id}/pagamentos?year=2026
     */
    public function studentPayments(Request $request, string $id): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        if (!$guardian->hasStudent($id)) {
            return response()->json(['status' => false, 'message' => 'Acesso negado'], 403);
        }

        $student = User::find($id);
        if (!$student) {
            return response()->json(['status' => false, 'message' => 'Estudante nao encontrado'], 404);
        }

        $year = $request->query('year', now()->year);

        try {
            $feeService = app(\App\Services\FeeCalculationService::class);
            $summary = $feeService->calculateYearSummary($student, (int) $year);

            // Buscar TODAS as referências pendentes do estudante (matrículas, taxas extras, etc.)
            // Apenas referências aprovadas aparecem no mobile (manuais precisam aprovação do admin)
            $allPendingReferences = \App\Models\PaymentReference::where('student_id', $student->id)
                ->where('fee_year', $year)
                ->where('status', 'pending')
                ->where('approval_status', 'approved')
                ->orderBy('created_at', 'desc')
                ->get();

            // Buscar referências pagas do ano
            $allPaidReferences = \App\Models\PaymentReference::where('student_id', $student->id)
                ->where('fee_year', $year)
                ->where('status', 'paid')
                ->orderBy('paid_at', 'desc')
                ->get();

            // Mapear meses com pending_reference (como a API do estudante)
            $monthsData = array_map(function ($month) use ($student) {
                $pendingReference = \App\Models\PaymentReference::where([
                    'student_id' => $student->id,
                    'fee_month' => $month['month'],
                    'fee_year' => $month['year'],
                    'status' => 'pending',
                    'approval_status' => 'approved',
                ])
                ->where('expires_at', '>', now())
                ->first();

                return [
                    'month'          => $month['month'],
                    'year'           => $month['year'],
                    'base_amount'    => (float) $month['base_amount'],
                    'fine_amount'    => (float) $month['fine_amount'],
                    'discount_applied' => (float) ($month['discount_applied'] ?? 0),
                    'total_expected' => (float) $month['total_expected'],
                    'is_paid'        => (bool) $month['is_paid'],
                    'paid_amount'    => (float) ($month['paid_amount'] ?? 0),
                    'is_overdue'     => (bool) $month['is_overdue'],
                    'due_date'       => $month['due_date']?->format('Y-m-d'),
                    'payment_date'   => $month['payment_date']?->format('Y-m-d H:i:s'),
                    'fee_names'      => $month['fee_names'] ?? 'Taxa Escolar',
                    'data_status'    => $month['is_paid'] ? 'paid' : ($month['is_overdue'] ? 'overdue' : 'pending'),
                    'status'         => $month['is_paid'] ? 'paid' : ($month['is_overdue'] ? 'overdue' : 'pending'),
                    'payment_id'     => $month['existing_payment']?->id ?? null,
                    'has_receipt'    => !empty($month['existing_payment']),
                    'pending_reference' => $pendingReference ? [
                        'id' => $pendingReference->id,
                        'reference_number' => $pendingReference->reference_number,
                        'entity_code' => $pendingReference->entity_code,
                        'amount' => (float) $pendingReference->amount,
                        'expires_at' => \Carbon\Carbon::parse($pendingReference->expires_at)->format('Y-m-d H:i:s'),
                        'days_until_expiry' => now()->diffInDays(\Carbon\Carbon::parse($pendingReference->expires_at), false),
                    ] : null,
                ];
            }, $summary['months_detail']);

            // Formatar referências pendentes (todas — incluindo matrículas, taxas extras)
            $formatReference = function ($ref) {
                $metadata = is_array($ref->metadata) ? $ref->metadata : [];
                $description = $metadata['description'] ?? null;
                $isExpired = $ref->expires_at && Carbon::parse($ref->expires_at)->isPast();

                return [
                    'id' => $ref->id,
                    'reference_number' => $ref->reference_number,
                    'entity_code' => $ref->entity_code,
                    'amount' => (float) $ref->amount,
                    'fine_amount' => (float) ($ref->fine_amount ?? 0),
                    'fee_month' => $ref->fee_month,
                    'fee_year' => $ref->fee_year,
                    'description' => $description,
                    'status' => $isExpired ? 'expired' : $ref->status,
                    'is_expired' => $isExpired,
                    'expires_at' => $ref->expires_at ? Carbon::parse($ref->expires_at)->format('Y-m-d H:i:s') : null,
                    'days_until_expiry' => $ref->expires_at ? (int) now()->diffInDays(Carbon::parse($ref->expires_at), false) : null,
                    'paid_at' => $ref->paid_at ? Carbon::parse($ref->paid_at)->format('Y-m-d H:i:s') : null,
                    'created_at' => $ref->created_at->format('Y-m-d H:i:s'),
                ];
            };

            // Instruções de pagamento dinâmicas do config
            $entity = config('payments.entity', '90013');
            $instructions = config('payments.payment_instructions', []);
            $paymentInstructions = [];
            foreach ($instructions as $key => $instruction) {
                $paymentInstructions[] = [
                    'method' => $instruction['title'] ?? $key,
                    'steps' => str_replace(['{entity}'], [$entity], $instruction['steps'] ?? ''),
                ];
            }

            return response()->json([
                'status' => true,
                'data'   => [
                    'student' => [
                        'id'         => $student->id,
                        'student_id' => $student->student_id,
                        'name'       => $student->name,
                        'class'      => $student->class->class ?? null,
                        'classroom'  => $student->classroom->name ?? null,
                    ],
                    'year'    => $year,
                    'summary' => [
                        'total_expected' => $summary['total_expected'],
                        'total_paid'     => $summary['total_paid'],
                        'total_pending'  => $summary['total_pending'],
                        'total_overdue'  => $summary['total_overdue'],
                        'total_fines'    => $summary['total_fines'],
                        'total_discounts' => $summary['total_discounts'] ?? 0,
                    ],
                    'months' => $monthsData,
                    'pending_references' => $allPendingReferences->map($formatReference)->values(),
                    'paid_references' => $allPaidReferences->map($formatReference)->values(),
                    'payment_instructions' => $paymentInstructions,
                    'entity_code' => $entity,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status'  => false,
                'message' => 'Erro ao carregar pagamentos.',
            ], 500);
        }
    }

    /**
     * Presencas de um estudante
     * GET /api/guardian/student/{id}/presencas?month=2&year=2026
     */
    public function studentAttendance(Request $request, string $id): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        if (!$guardian->hasStudent($id)) {
            return response()->json(['status' => false, 'message' => 'Acesso negado'], 403);
        }

        $student = User::with('classroom')->find($id);
        $month = $request->query('month', now()->month);
        $year  = $request->query('year', now()->year);

        $attendances = Attendance::where('student_id', $id)
            ->whereMonth('date', $month)
            ->whereYear('date', $year)
            ->orderBy('date', 'desc')
            ->get();

        $total     = $attendances->count();
        $present   = $attendances->where('status', 'present')->count();
        $absent    = $attendances->where('status', 'absent')->count();
        $late      = $attendances->where('status', 'late')->count();
        $justified = $attendances->where('status', 'justified')->count();
        $percentage = $total > 0 ? round(($present + $late + $justified) / $total * 100, 1) : 0;

        return response()->json([
            'status'  => true,
            'student' => [
                'name'       => $student->name ?? null,
                'student_id' => $student->student_id ?? null,
                'classroom'  => $student->classroom?->name,
            ],
            'month'   => (int) $month,
            'year'    => (int) $year,
            'summary' => [
                'total'      => $total,
                'present'    => $present,
                'absent'     => $absent,
                'late'       => $late,
                'justified'  => $justified,
                'percentage' => $percentage,
            ],
            'records' => $attendances->map(function ($a) {
                return [
                    'date'        => $a->date->format('Y-m-d'),
                    'day_name'    => $a->date->translatedFormat('l'),
                    'subject'     => $a->subject,
                    'status'      => $a->status,
                    'status_label' => Attendance::statuses()[$a->status] ?? $a->status,
                    'observation' => $a->observation,
                ];
            })->values(),
        ], 200);
    }

    /**
     * Comunicados relevantes para os estudantes do encarregado
     * GET /api/guardian/comunicados
     */
    public function comunicados(Request $request): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        $students = $guardian->students()->with('classroom')->get();
        $classroomIds = $students->pluck('classroom_id')->unique()->toArray();
        $studentIds   = $students->pluck('id')->toArray();

        $comunicados = Noticeboard::where('type', 'all')
            ->orWhereIn('class_id', $classroomIds)
            ->orWhereIn('student_id', $studentIds)
            ->orderByDesc('created_at')
            ->get();

        return response()->json([
            'status'       => true,
            'comunicados'  => $comunicados->map(function ($c) {
                return [
                    'id'         => $c->id,
                    'title'      => $c->title,
                    'message'    => $c->message,
                    'type'       => $c->type,
                    'status'     => $c->status,
                    'created_at' => $c->created_at?->format('Y-m-d H:i'),
                ];
            }),
            'total' => $comunicados->count(),
        ], 200);
    }

    /**
     * Matrizes/documentos disponiveis para os estudantes do encarregado
     * GET /api/guardian/matrizes
     */
    public function matrizes(Request $request): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        $students = $guardian->students()->with('classroom')->get();
        $classroomIds = $students->pluck('classroom_id')->unique()->toArray();

        $downloads = Downloadsetings::where(function ($query) use ($classroomIds) {
            $query->where('type', 'all')
                  ->whereIn('class_id', $classroomIds);
        })
        ->orWhereIn('class_id', $classroomIds)
        ->orderByDesc('created_at')
        ->get();

        return response()->json([
            'status'   => true,
            'matrizes' => $downloads->map(function ($d) {
                return [
                    'id'         => $d->id,
                    'title'      => $d->title,
                    'type'       => $d->type,
                    'class_id'   => $d->class_id,
                    'created_at' => $d->created_at?->format('Y-m-d H:i'),
                ];
            }),
        ], 200);
    }

    /**
     * Perfil do encarregado
     * GET /api/guardian/profile
     */
    public function profile(Request $request): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        $studentsCount = $guardian->students()->count();

        return response()->json([
            'status'   => true,
            'guardian' => [
                'id'             => $guardian->id,
                'name'           => $guardian->name,
                'phone'          => $guardian->phone,
                'email'          => $guardian->email,
                'relationship'   => $guardian->relationship,
                'students_count' => $studentsCount,
                'last_login_at'  => $guardian->last_login_at?->format('Y-m-d H:i'),
            ],
        ], 200);
    }

    /**
     * Alterar senha
     * POST /api/guardian/change-password
     * Body: { current_password, new_password, new_password_confirmation }
     */
    public function changePassword(Request $request): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        $validator = Validator::make($request->all(), [
            'current_password' => 'required',
            'new_password'     => 'required|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
        }

        if (!Hash::check($request->current_password, $guardian->password)) {
            return response()->json(['status' => false, 'error' => 'Senha actual incorrecta.'], 400);
        }

        $guardian->update(['password' => Hash::make($request->new_password)]);

        return response()->json(['status' => true, 'message' => 'Senha alterada com sucesso.'], 200);
    }

    /**
     * Registar device token para push notifications
     * POST /api/guardian/device-token
     * Body: { device_token, device_model? }
     */
    public function storeDeviceToken(Request $request): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        $validator = Validator::make($request->all(), [
            'device_token' => 'required|string',
            'device_model' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
        }

        $existing = $guardian->deviceTokens()->where('token', $request->device_token)->first();

        if ($existing) {
            $existing->update([
                'device_model' => $request->device_model ?? $existing->device_model,
            ]);
        } else {
            $guardian->deviceTokens()->create([
                'token'        => $request->device_token,
                'device_model' => $request->device_model ?? 'Desconhecido',
            ]);
        }

        return response()->json(['status' => true, 'message' => 'Token registado.'], 200);
    }

    /**
     * Remover device token
     * POST /api/guardian/remove-device-token
     * Body: { device_token }
     */
    public function removeDeviceToken(Request $request): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        $deleted = $guardian->deviceTokens()->where('token', $request->device_token)->delete();

        return response()->json([
            'status'  => true,
            'message' => $deleted ? 'Token removido.' : 'Token nao encontrado.',
        ], 200);
    }

    /**
     * Gerar referência de pagamento para um estudante
     * POST /api/guardian/student/{id}/generate-reference
     * Body: { month, year, amount }
     */
    public function generateReference(Request $request, string $id): JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        if (!$guardian->hasStudent($id)) {
            return response()->json(['status' => false, 'message' => 'Acesso negado'], 403);
        }

        $student = User::find($id);
        if (!$student) {
            return response()->json(['status' => false, 'message' => 'Estudante não encontrado'], 404);
        }

        $monthNumber = (int) $request->month;
        $year = (int) $request->year;

        // Converter número do mês para nome em português
        $monthNames = [
            1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
            5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
            9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
        ];
        $monthName = $monthNames[$monthNumber] ?? $request->month;

        try {
            // Verificar se já existe referência pendente
            $existingReference = \App\Models\PaymentReference::where([
                'student_id' => $student->id,
                'fee_month' => $monthName,
                'fee_year' => $year,
                'status' => 'pending'
            ])->where('expires_at', '>', now())->first();

            if ($existingReference) {
                return response()->json([
                    'status' => true,
                    'message' => 'Referência existente ainda válida',
                    'data' => [
                        'id' => $existingReference->id,
                        'entity_code' => $existingReference->entity_code,
                        'reference_number' => $existingReference->reference_number,
                        'amount' => (float) $existingReference->amount,
                        'fee_month' => $existingReference->fee_month,
                        'fee_year' => $existingReference->fee_year,
                        'expires_at' => Carbon::parse($existingReference->expires_at)->format('Y-m-d H:i:s'),
                        'days_until_expiry' => now()->diffInDays(Carbon::parse($existingReference->expires_at), false),
                    ]
                ], 200);
            }

            // Calcular fee
            $feeService = app(\App\Services\FeeCalculationService::class);
            $calculation = $feeService->calculateFeeForStudent($student, $monthName, $year);

            $baseAmount = $calculation['base_amount'];
            $fineAmount = $calculation['fine_amount'] ?? 0;
            $totalAmount = $baseAmount + $fineAmount;

            if ($totalAmount <= 0) {
                return response()->json([
                    'status' => false,
                    'message' => 'Valor da taxa inválido para este mês'
                ], 400);
            }

            $entity = config('payments.entity', '90013');
            $expiresAt = \App\Models\FeeMonthDeadline::calculateExpiresAt($monthName, $year);

            $referenceGenerator = app(\App\Services\ReferenceGenerator::class);
            $reference = $referenceGenerator->makeV1FromStudent($monthName, $student, $totalAmount);

            $paymentReference = \App\Models\PaymentReference::create([
                'student_id' => $student->id,
                'entity_code' => $entity,
                'reference_number' => $reference,
                'amount' => $totalAmount,
                'fine_amount' => $fineAmount,
                'fee_month' => $monthName,
                'fee_year' => $year,
                'expires_at' => $expiresAt,
                'status' => 'pending',
                'approval_status' => 'approved',
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Referência gerada com sucesso',
                'data' => [
                    'id' => $paymentReference->id,
                    'entity_code' => $paymentReference->entity_code,
                    'reference_number' => $paymentReference->reference_number,
                    'amount' => (float) $paymentReference->amount,
                    'fee_month' => $paymentReference->fee_month,
                    'fee_year' => $paymentReference->fee_year,
                    'expires_at' => Carbon::parse($expiresAt)->format('Y-m-d H:i:s'),
                    'days_until_expiry' => now()->diffInDays(Carbon::parse($expiresAt), false),
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Guardian generate reference error', [
                'guardian_id' => $guardian->id,
                'student_id' => $id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => false,
                'message' => 'Erro ao gerar referência: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download recibo de pagamento de um estudante
     * GET /api/guardian/student/{studentId}/receipt/{paymentId}/download
     */
    public function downloadStudentReceipt(Request $request, string $studentId, int $paymentId): \Illuminate\Http\Response|JsonResponse
    {
        $guardian = $this->guardianOrFail();
        if (!$guardian) {
            return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
        }

        if (!$guardian->hasStudent($studentId)) {
            return response()->json(['status' => false, 'message' => 'Acesso negado'], 403);
        }

        $student = User::with(['class', 'classroom'])->find($studentId);
        if (!$student) {
            return response()->json(['status' => false, 'message' => 'Estudante não encontrado'], 404);
        }

        $payment = \App\Models\Fee_assign::where('id', $paymentId)
            ->where('student_id', $studentId)
            ->first();

        if (!$payment) {
            return response()->json(['status' => false, 'message' => 'Pagamento não encontrado'], 404);
        }

        try {
            $feeService = app(\App\Services\FeeCalculationService::class);
            $calculation = $feeService->calculateFeeForStudent(
                $student,
                $payment->month,
                (int) $payment->year
            );

            $paidAt = $payment->created_at ?? now();
            $receiptNumber = str_pad($payment->id, 6, '0', STR_PAD_LEFT) . '/' . $paidAt->format('Y');

            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.receipt-official', [
                'payment' => $payment,
                'student' => $student,
                'amount_paid' => $payment->amount + $payment->fine - $payment->discount,
                'base_amount' => $payment->amount,
                'fine_amount' => $payment->fine,
                'discount_amount' => $payment->discount,
                'receipt_number' => $receiptNumber,
                'paid_at' => $paidAt,
                'transaction_id' => $payment->transaction_id,
                'reference_formatted' => $payment->reference_number,
                'payment_method' => $payment->payment_mode ?? 'Referência',
                'notes' => null,
            ]);

            $pdf->setPaper('a4');

            return response($pdf->output(), 200)
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="recibo_' . $receiptNumber . '.pdf"');

        } catch (\Exception $e) {
            Log::error('Guardian receipt download error', [
                'guardian_id' => $guardian->id,
                'student_id' => $studentId,
                'payment_id' => $paymentId,
                'error' => $e->getMessage()
            ]);

            return response()->json(['status' => false, 'message' => 'Erro ao gerar recibo'], 500);
        }
    }
}
