<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\PaymentReference;
use App\Models\User;
use App\Services\PaymentReconciliationService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReconciliationController extends Controller
{
    protected $reconciliationService;

    public function __construct(PaymentReconciliationService $reconciliationService)
    {
        $this->reconciliationService = $reconciliationService;
    }

    /**
     * Dashboard de reconciliação
     */
    public function dashboard()
    {
        // Estatísticas gerais
        $stats = [
            'today' => $this->getTodayStats(),
            'week' => $this->getWeekStats(),
            'month' => $this->getMonthStats(),
            'unmatched' => $this->getUnmatchedPayments(),
            'pending_references' => $this->getPendingReferences()
        ];

        // Últimos logs de reconciliação
        $recentLogs = DB::table('reconciliation_logs')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Pagamentos não correspondidos
        $unmatchedPayments = DB::table('unmatched_payments')
            ->where('status', 'unmatched')
            ->orderBy('payment_date', 'desc')
            ->paginate(20);

        return view('admin.reconciliation.dashboard', compact('stats', 'recentLogs', 'unmatchedPayments'));
    }

    /**
     * Resolver pagamento não correspondido
     */
    public function resolveUnmatched(Request $request, $id)
    {
        $request->validate([
            'student_id' => 'required|exists:users,id',
            'month' => 'required|string',
            'year' => 'required|integer',
            'action' => 'required|in:assign,refund,ignore'
        ]);

        DB::beginTransaction();
        
        try {
            $unmatched = DB::table('unmatched_payments')->find($id);
            
            if (!$unmatched) {
                throw new \Exception('Pagamento não encontrado');
            }

            switch ($request->action) {
                case 'assign':
                    $this->assignPaymentToStudent($unmatched, $request);
                    break;
                    
                case 'refund':
                    $this->markForRefund($unmatched);
                    break;
                    
                case 'ignore':
                    $this->ignorePayment($unmatched);
                    break;
            }

            // Atualizar status
            DB::table('unmatched_payments')
                ->where('id', $id)
                ->update([
                    'status' => 'resolved',
                    'resolved_by' => auth()->id(),
                    'resolved_at' => now(),
                    'notes' => $request->notes
                ]);

            DB::commit();
            
            return redirect()->back()->with('success', 'Pagamento resolvido com sucesso');
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to resolve unmatched payment', [
                'id' => $id,
                'error' => $e->getMessage()
            ]);
            
            return redirect()->back()->with('error', 'Erro ao resolver pagamento: ' . $e->getMessage());
        }
    }

    /**
     * Atribuir pagamento a estudante
     */
    private function assignPaymentToStudent($unmatched, $request)
    {
        $student = User::find($request->student_id);
        
        // Criar referência retroativa
        $reference = PaymentReference::create([
            'student_id' => $student->id,
            'entity_code' => config('bmeps.entity_id'),
            'reference_number' => $unmatched->reference,
            'amount' => $unmatched->amount,
            'fee_month' => $request->month,
            'fee_year' => $request->year,
            'status' => 'paid',
            'paid_at' => $unmatched->payment_date,
            'transaction_id' => $unmatched->transaction_id,
            'terminal_id' => $unmatched->terminal_id,
            'expires_at' => Carbon::parse($unmatched->payment_date)->addDays(30)
        ]);

        // Criar registo em Fee_assign
        \App\Models\Fee_assign::create([
            'student_id' => $student->id,
            'amount' => $unmatched->amount,
            'discount' => 0,
            'fine' => 0,
            'payment_mode' => 'Reference',
            'note' => "Pagamento reconciliado manualmente - Ref: {$unmatched->reference}",
            'pay_day' => Carbon::parse($unmatched->payment_date)->day,
            'pay_month' => Carbon::parse($unmatched->payment_date)->month,
            'pay_year' => Carbon::parse($unmatched->payment_date)->year,
            'pay_type' => 'reference',
            'status' => 'Paid',
            'month' => $request->month,
            'year' => $request->year,
            'transaction_id' => $unmatched->transaction_id,
            'reference_number' => $unmatched->reference,
            'payment_date' => $unmatched->payment_date
        ]);

        Log::info('Unmatched payment manually assigned', [
            'transaction_id' => $unmatched->transaction_id,
            'student_id' => $student->id,
            'resolved_by' => auth()->id()
        ]);
    }

    /**
     * Marcar para reembolso
     */
    private function markForRefund($unmatched)
    {
        DB::table('payment_refunds')->insert([
            'transaction_id' => $unmatched->transaction_id,
            'reference' => $unmatched->reference,
            'amount' => $unmatched->amount,
            'payment_date' => $unmatched->payment_date,
            'status' => 'pending',
            'requested_by' => auth()->id(),
            'requested_at' => now(),
            'created_at' => now(),
            'updated_at' => now()
        ]);

        Log::info('Payment marked for refund', [
            'transaction_id' => $unmatched->transaction_id,
            'amount' => $unmatched->amount
        ]);
    }

    /**
     * Ignorar pagamento
     */
    private function ignorePayment($unmatched)
    {
        Log::info('Payment ignored', [
            'transaction_id' => $unmatched->transaction_id,
            'ignored_by' => auth()->id()
        ]);
    }

    /**
     * Forçar reconciliação manual
     */
    public function forceReconciliation(Request $request)
    {
        try {
            $stats = $this->reconciliationService->processAllPendingFiles();
            
            return response()->json([
                'success' => true,
                'message' => 'Reconciliação executada com sucesso',
                'stats' => $stats
            ]);
            
        } catch (\Exception $e) {
            Log::error('Manual reconciliation failed', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Erro na reconciliação: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Buscar estudante por referência
     */
    public function searchStudentByReference(Request $request)
    {
        $reference = $request->get('reference');
        
        // Tentar extrair ID do estudante da referência (primeiros 7 dígitos)
        $possibleStudentCode = substr($reference, 0, 7);
        
        // Buscar estudantes possíveis
        $students = User::where('role', 'student')
            ->where(function($query) use ($possibleStudentCode, $reference) {
                $query->where('student_id', 'LIKE', '%' . $possibleStudentCode . '%')
                      ->orWhere('id', 'LIKE', '%' . $possibleStudentCode . '%');
            })
            ->select('id', 'name', 'student_id', 'email')
            ->limit(10)
            ->get();

        return response()->json($students);
    }

    /**
     * Obter estatísticas do dia
     */
    private function getTodayStats()
    {
        return DB::table('payment_reconciliations')
            ->whereDate('processed_at', Carbon::today())
            ->selectRaw('COUNT(*) as count, SUM(amount) as total')
            ->first();
    }

    /**
     * Obter estatísticas da semana
     */
    private function getWeekStats()
    {
        return DB::table('payment_reconciliations')
            ->whereBetween('processed_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()])
            ->selectRaw('COUNT(*) as count, SUM(amount) as total')
            ->first();
    }

    /**
     * Obter estatísticas do mês
     */
    private function getMonthStats()
    {
        return DB::table('payment_reconciliations')
            ->whereMonth('processed_at', Carbon::now()->month)
            ->whereYear('processed_at', Carbon::now()->year)
            ->selectRaw('COUNT(*) as count, SUM(amount) as total')
            ->first();
    }

    /**
     * Obter pagamentos não correspondidos
     */
    private function getUnmatchedPayments()
    {
        return DB::table('unmatched_payments')
            ->where('status', 'unmatched')
            ->selectRaw('COUNT(*) as count, SUM(amount) as total')
            ->first();
    }

    /**
     * Obter referências pendentes
     */
    private function getPendingReferences()
    {
        return PaymentReference::where('status', 'pending')
            ->where('expires_at', '>', now())
            ->selectRaw('COUNT(*) as count, SUM(amount) as total')
            ->first();
    }

    /**
     * Exportar relatório de reconciliação
     */
    public function exportReport(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date'
        ]);

        $data = DB::table('payment_reconciliations')
            ->join('payment_references', 'payment_reconciliations.reference', '=', 'payment_references.reference_number')
            ->join('users', 'payment_references.student_id', '=', 'users.id')
            ->whereBetween('payment_reconciliations.processed_at', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ])
            ->select(
                'payment_reconciliations.*',
                'users.name as student_name',
                'users.student_id',
                'payment_references.fee_month',
                'payment_references.fee_year'
            )
            ->get();

        // Gerar CSV
        $filename = 'reconciliation_report_' . now()->format('Y-m-d_His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($data) {
            $file = fopen('php://output', 'w');
            
            // Header
            fputcsv($file, [
                'Transaction ID',
                'Reference',
                'Student ID',
                'Student Name',
                'Month',
                'Year',
                'Amount',
                'Payment Date',
                'Terminal ID',
                'Processed At'
            ]);
            
            // Data
            foreach ($data as $row) {
                fputcsv($file, [
                    $row->transaction_id,
                    $row->reference,
                    $row->student_id,
                    $row->student_name,
                    $row->fee_month,
                    $row->fee_year,
                    $row->amount,
                    $row->payment_date,
                    $row->terminal_id,
                    $row->processed_at
                ]);
            }
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
