<?php

namespace App\Http\Livewire\Admin\Attendance;

use App\Models\Attendance;
use App\Models\Classroom;
use App\Models\TeacherSubject;
use App\Models\User;
use App\Models\Admin;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;

class MarkAttendance extends Component
{
    use FindGuard, LivewireAlert;

    public $year;
    public $classroom_id;
    public $date;
    public $students = [];
    public $attendanceData = [];
    public $loadedStudents = false;
    public $isTeacher = false;

    protected $listeners = ['confirmed', 'cancelled'];

    public function mount()
    {
        $this->year = date('Y');
        $this->date = date('Y-m-d');
        $this->user = $this->findGuardType()->user();
        $this->isTeacher = ($this->user->role == 2);
    }

    public function updatedYear()
    {
        $this->classroom_id = null;
        $this->students = [];
        $this->attendanceData = [];
        $this->loadedStudents = false;
    }

    public function updatedClassroomId()
    {
        $this->students = [];
        $this->attendanceData = [];
        $this->loadedStudents = false;
    }

    public function updatedDate()
    {
        if ($this->classroom_id) {
            $this->loadStudents();
        }
    }

    public function loadStudents()
    {
        $this->validate([
            'classroom_id' => 'required',
            'date' => 'required|date',
            'year' => 'required',
        ]);

        $this->students = User::where('classroom_id', $this->classroom_id)
            ->where('year_edu', $this->year)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // Load existing attendance for this date
        $existing = Attendance::where('classroom_id', $this->classroom_id)
            ->where('date', $this->date)
            ->where('year', $this->year)
            ->pluck('status', 'student_id')
            ->toArray();

        $existingObs = Attendance::where('classroom_id', $this->classroom_id)
            ->where('date', $this->date)
            ->where('year', $this->year)
            ->pluck('observation', 'student_id')
            ->toArray();

        foreach ($this->students as $student) {
            $this->attendanceData[$student->id] = [
                'status' => $existing[$student->id] ?? 'present',
                'observation' => $existingObs[$student->id] ?? '',
            ];
        }

        $this->loadedStudents = true;
    }

    public function saveAttendance()
    {
        $this->confirm('Tem certeza de que deseja salvar a presença?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, salvar!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        $user = $this->findGuardType()->user();
        $savedCount = 0;

        DB::beginTransaction();

        try {
            foreach ($this->attendanceData as $studentId => $data) {
                Attendance::updateOrCreate(
                    [
                        'student_id' => $studentId,
                        'classroom_id' => $this->classroom_id,
                        'date' => $this->date,
                    ],
                    [
                        'status' => $data['status'],
                        'observation' => $data['observation'] ?: null,
                        'recorded_by' => $user->id,
                        'year' => $this->year,
                    ]
                );
                $savedCount++;
            }

            DB::commit();

            Note::createSystemNotification(Admin::class, 'Presença', "Presença registada para {$savedCount} estudantes");
            $this->emit('noteAdded');
            $this->alert('success', "Presença salva com sucesso! ({$savedCount} estudantes)");

        } catch (\Exception $e) {
            DB::rollBack();
            $this->alert('error', 'Erro ao salvar presença: ' . $e->getMessage());
        }
    }

    public function cancelled()
    {
        $this->alert('error', 'Operação cancelada.');
    }

    public function markAllAs($status)
    {
        foreach ($this->attendanceData as $studentId => $data) {
            $this->attendanceData[$studentId]['status'] = $status;
        }
    }

    public function render()
    {
        $user = $this->findGuardType()->user();
        $this->isTeacher = ($user->role == 2);

        $classrooms = collect();

        if ($this->isTeacher) {
            $classroomIds = TeacherSubject::where('teacher_id', $user->id)
                ->whereNotNull('classroom_id')
                ->distinct()
                ->pluck('classroom_id');

            $classrooms = Classroom::whereIn('id', $classroomIds)
                ->orderBy('class')
                ->orderBy('name')
                ->get();
        } else {
            $classrooms = Classroom::orderBy('class')->orderBy('name')->get();
        }

        return view('livewire.admin.attendance.mark-attendance', [
            'classrooms' => $classrooms,
        ]);
    }
}
