<?php

namespace App\Http\Livewire\Admin\Fee;

use App\Models\PaymentReference;
use App\Models\Classroom;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Services\SmsService;

class ApproveReferences extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    // Filtros
    public $filterStatus = 'pending_approval';
    public $filterClass;
    public $filterTurma;
    public $filterMonth;
    public $filterYear;
    public $searchTerm = '';

    // Aprovação em massa
    public $selectedReferences = [];
    public $selectAll = false;

    // Modal de aprovação/rejeição em massa
    public $showBulkActionModal = false;
    public $bulkAction = 'approve'; // approve ou reject
    public $bulkNotes = '';
    public $sendBulkSms = true;

    // Modal de aprovação individual
    public $showApprovalModal = false;
    public $approvalReferenceId;
    public $approvalAction = 'approve'; // approve ou reject
    public $approvalNotes;
    public $sendSmsAfterApproval = true;

    protected $listeners = ['refreshComponent' => '$refresh'];

    public function mount()
    {
        $this->filterYear = date('Y');
    }

    /**
     * Verifica se o usuário atual é PCA (role 0)
     */
    private function isPca()
    {
        return auth()->guard('admin')->user()->role == 0;
    }

    public function updatedSelectAll($value)
    {
        if ($value) {
            // Pega apenas os IDs da página atual para evitar problemas de serialização
            $this->selectedReferences = PaymentReference::where('approval_status', $this->filterStatus)
                ->when($this->filterClass, fn($q) => $q->where('classe', $this->filterClass))
                ->when($this->filterTurma, fn($q) => $q->where('turma', $this->filterTurma))
                ->when($this->filterMonth, fn($q) => $q->where('fee_month', $this->filterMonth))
                ->when($this->filterYear, fn($q) => $q->where('fee_year', $this->filterYear))
                ->when($this->searchTerm, function($q) {
                    $q->where(function ($sq) {
                        $sq->where('reference_number', 'like', '%' . $this->searchTerm . '%')
                          ->orWhereHas('student', function ($ssq) {
                              $ssq->where('name', 'like', '%' . $this->searchTerm . '%')
                                ->orWhere('student_id', 'like', '%' . $this->searchTerm . '%');
                          });
                    });
                })
                ->pluck('id')
                ->toArray();
        } else {
            $this->selectedReferences = [];
        }
    }

    public function openApprovalModal($referenceId, $action = 'approve')
    {
        $this->approvalReferenceId = $referenceId;
        $this->approvalAction = $action;
        $this->approvalNotes = '';
        $this->sendSmsAfterApproval = ($action === 'approve');
        $this->showApprovalModal = true;
    }

    public function processIndividualApproval()
    {
        // Apenas PCA pode aprovar/rejeitar
        if (!$this->isPca()) {
            session()->flash('error', 'Acesso negado. Apenas o PCA pode aprovar ou rejeitar referências.');
            return;
        }

        try {
            $reference = PaymentReference::with('student')->findOrFail($this->approvalReferenceId);

            DB::beginTransaction();

            $smsSent = false;

            if ($this->approvalAction === 'approve') {
                $reference->approve($this->approvalNotes);

                if ($this->sendSmsAfterApproval && $reference->student) {
                    $smsSent = $this->sendApprovalSms($reference);
                }

                $successMessage = "Referência {$reference->reference_number} aprovada com sucesso!";
                if ($this->sendSmsAfterApproval && !$smsSent) {
                    $successMessage .= " (SMS não enviado - verifique se o estudante tem número de telemóvel)";
                }
                session()->flash('success', $successMessage);
            } else {
                $reference->reject($this->approvalNotes);
                session()->flash('success', "Referência {$reference->reference_number} rejeitada.");
            }

            DB::commit();

            Log::info('Reference approval processed', [
                'reference_id' => $reference->id,
                'reference_number' => $reference->reference_number,
                'action' => $this->approvalAction,
                'approved_by' => auth()->id(),
                'sms_sent' => $smsSent,
            ]);

            $this->showApprovalModal = false;
            $this->reset(['approvalReferenceId', 'approvalAction', 'approvalNotes']);

        } catch (\Exception $e) {
            DB::rollback();

            Log::error('Error processing reference approval', [
                'reference_id' => $this->approvalReferenceId,
                'error' => $e->getMessage(),
            ]);

            session()->flash('error', 'Erro ao processar aprovação: ' . $e->getMessage());
        }
    }

    public function openBulkActionModal($action)
    {
        if (empty($this->selectedReferences)) {
            session()->flash('warning', 'Nenhuma referência selecionada.');
            return;
        }

        $this->bulkAction = $action;
        $this->bulkNotes = '';
        $this->sendBulkSms = ($action === 'approve');
        $this->showBulkActionModal = true;
    }

    public function processBulkAction()
    {
        // Apenas PCA pode aprovar/rejeitar
        if (!$this->isPca()) {
            session()->flash('error', 'Acesso negado. Apenas o PCA pode aprovar ou rejeitar referências.');
            return;
        }

        if (empty($this->selectedReferences)) {
            session()->flash('warning', 'Nenhuma referência selecionada.');
            return;
        }

        try {
            DB::beginTransaction();

            $references = PaymentReference::with('student')
                ->whereIn('id', $this->selectedReferences)
                ->where('approval_status', 'pending_approval')
                ->get();

            $processed = 0;
            $smsSent = 0;
            $notes = $this->bulkNotes ?: ($this->bulkAction === 'approve' ? 'Aprovação em massa' : 'Rejeição em massa');

            foreach ($references as $reference) {
                if ($this->bulkAction === 'approve') {
                    $reference->approve($notes);
                    $processed++;

                    if ($this->sendBulkSms && $reference->student) {
                        if ($this->sendApprovalSms($reference)) {
                            $smsSent++;
                        }
                    }
                } else {
                    $reference->reject($notes);
                    $processed++;
                }
            }

            DB::commit();

            if ($this->bulkAction === 'approve') {
                session()->flash('success', "$processed referências aprovadas com sucesso! SMS enviados: $smsSent");
            } else {
                session()->flash('success', "$processed referências rejeitadas.");
            }

            Log::info('Bulk action processed', [
                'action' => $this->bulkAction,
                'count' => $processed,
                'sms_sent' => $smsSent,
                'notes' => $notes,
                'processed_by' => auth()->id(),
            ]);

            $this->showBulkActionModal = false;
            $this->reset(['selectedReferences', 'selectAll', 'bulkNotes', 'bulkAction']);

        } catch (\Exception $e) {
            DB::rollback();

            Log::error('Error in bulk action', [
                'action' => $this->bulkAction,
                'error' => $e->getMessage(),
            ]);

            session()->flash('error', 'Erro na operação em massa: ' . $e->getMessage());
        }
    }

    public function approveSelected()
    {
        $this->openBulkActionModal('approve');
    }

    public function rejectSelected()
    {
        $this->openBulkActionModal('reject');
    }

    /**
     * Coloca uma referência de volta como pendente
     */
    public function resetToPending($referenceId)
    {
        try {
            $reference = PaymentReference::findOrFail($referenceId);

            // Só permite voltar a pendente se estiver aprovada ou rejeitada
            if (!in_array($reference->approval_status, ['approved', 'rejected'])) {
                session()->flash('warning', 'Esta referência não pode ser colocada como pendente.');
                return;
            }

            // Se já foi paga, não permite alterar
            if ($reference->status === 'paid') {
                session()->flash('error', 'Não é possível alterar o status de uma referência já paga.');
                return;
            }

            DB::beginTransaction();

            $reference->update([
                'approval_status' => 'pending_approval',
                'approved_at' => null,
                'approved_by' => null,
                'approval_notes' => null,
            ]);

            DB::commit();

            Log::info('Reference reset to pending', [
                'reference_id' => $reference->id,
                'reference_number' => $reference->reference_number,
                'reset_by' => auth()->id(),
            ]);

            session()->flash('success', "Referência {$reference->reference_number} colocada como pendente. Pode aprovar novamente.");

        } catch (\Exception $e) {
            DB::rollback();

            Log::error('Error resetting reference to pending', [
                'reference_id' => $referenceId,
                'error' => $e->getMessage(),
            ]);

            session()->flash('error', 'Erro ao alterar status: ' . $e->getMessage());
        }
    }

    private function sendApprovalSms(PaymentReference $reference)
    {
        try {
            if (!$reference->student || !$reference->student->mobile) {
                Log::warning('Cannot send SMS - missing mobile number', [
                    'reference_id' => $reference->id,
                    'student_id' => $reference->student_id
                ]);
                return false;
            }

            // fee_month pode ser número ou nome (em português ou inglês)
            $monthName = is_numeric($reference->fee_month)
                ? $this->getPortugueseMonth($reference->fee_month)
                : $this->getPortugueseMonthName($reference->fee_month);

            $expiresDate = $reference->expires_at ? $reference->expires_at->format('d/m/Y') : 'N/A';

            // Buscar descrição base de pagamento - primeiro tenta metadata, depois fee assigns
            $baseDescription = 'Pagamento de Mensalidade';

            // Tentar obter da metadata (novo sistema)
            if (!empty($reference->metadata['description'])) {
                $baseDescription = $reference->metadata['description'];
            }
            // Fallback para fee assigns (sistema antigo)
            elseif ($reference->feeAssigns()->exists()) {
                $feeAssigns = $reference->feeAssigns()->with('fee_master')->get();
                $descriptions = $feeAssigns->map(function($assign) {
                    return $assign->fee_master ? $assign->fee_master->name : null;
                })->filter()->unique();

                if ($descriptions->isNotEmpty()) {
                    $baseDescription = $descriptions->implode(', ');
                }
            }

            // Montar descrição completa com mês/ano se não estiver já incluído
            $description = $baseDescription;
            if (!str_contains($description, $monthName) && !str_contains($description, $reference->fee_year)) {
                $description .= ' - ' . $monthName . '/' . $reference->fee_year;
            }

            // Mensagem com Entidade, Referência, Valor e Descrição
            $message = sprintf(
                "COPMOZ - Referencia Aprovada!\n\nEstudante: %s\nDescricao: %s\nMes: %s/%s\n\nEntidade: %s\nReferencia: %s\nValor: %.2f MT\nExpira: %s\n\nPague via eBanking, Balcões, ATMs da RedeSIMO e pelo USSD - BCI",
                $reference->student->name,
                $description,
                $monthName,
                $reference->fee_year,
                $reference->entity_code,
                $reference->reference_number,
                $reference->amount,
                $expiresDate
            );

            // Usar SmsService para enviar SMS
            $smsService = new SmsService();
            $result = $smsService->send($reference->student->mobile, $message);

            if ($result['success']) {
                Log::info('Approval SMS sent', [
                    'reference_id' => $reference->id,
                    'student_id' => $reference->student_id,
                    'mobile' => $reference->student->mobile,
                    'gateway' => $result['gateway'],
                    'message_id' => $result['message_id'] ?? null,
                    'entity' => $reference->entity_code,
                    'reference' => $reference->reference_number,
                    'amount' => $reference->amount,
                    'description' => $description,
                ]);

                return true;
            } else {
                Log::error('Error sending approval SMS', [
                    'reference_id' => $reference->id,
                    'error' => $result['error'] ?? 'Unknown error',
                    'gateway' => $result['gateway'],
                ]);
                return false;
            }

        } catch (\Exception $e) {
            Log::error('Error sending approval SMS', [
                'reference_id' => $reference->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    private function getPortugueseMonth($monthNumber)
    {
        $months = [
            1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
            5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
            9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
        ];

        return $months[$monthNumber] ?? $monthNumber;
    }

    private function getPortugueseMonthName($monthName)
    {
        // Mapeamento português → português (capitalizado)
        $portugueseMonths = [
            'janeiro' => 'Janeiro', 'fevereiro' => 'Fevereiro', 'março' => 'Março',
            'abril' => 'Abril', 'maio' => 'Maio', 'junho' => 'Junho',
            'julho' => 'Julho', 'agosto' => 'Agosto', 'setembro' => 'Setembro',
            'outubro' => 'Outubro', 'novembro' => 'Novembro', 'dezembro' => 'Dezembro',
        ];

        // Mapeamento inglês → português
        $englishToPortuguese = [
            'january' => 'Janeiro', 'february' => 'Fevereiro', 'march' => 'Março',
            'april' => 'Abril', 'may' => 'Maio', 'june' => 'Junho',
            'july' => 'Julho', 'august' => 'Agosto', 'september' => 'Setembro',
            'october' => 'Outubro', 'november' => 'Novembro', 'december' => 'Dezembro',
        ];

        $monthLower = strtolower($monthName);

        // Tentar português primeiro
        if (isset($portugueseMonths[$monthLower])) {
            return $portugueseMonths[$monthLower];
        }

        // Tentar inglês
        if (isset($englishToPortuguese[$monthLower])) {
            return $englishToPortuguese[$monthLower];
        }

        // Fallback: capitalizar primeira letra
        return ucfirst($monthName);
    }

    private function getReferencesQuery()
    {
        return PaymentReference::with(['student', 'approver'])
            ->where('approval_status', $this->filterStatus)
            ->when($this->filterClass, fn($q) => $q->where('classe', $this->filterClass))
            ->when($this->filterTurma, fn($q) => $q->where('turma', $this->filterTurma))
            ->when($this->filterMonth, fn($q) => $q->where('fee_month', $this->filterMonth))
            ->when($this->filterYear, fn($q) => $q->where('fee_year', $this->filterYear))
            ->when($this->searchTerm, function($q) {
                $q->where(function ($sq) {
                    $sq->where('reference_number', 'like', '%' . $this->searchTerm . '%')
                      ->orWhereHas('student', function ($ssq) {
                          $ssq->where('name', 'like', '%' . $this->searchTerm . '%')
                            ->orWhere('student_id', 'like', '%' . $this->searchTerm . '%');
                      });
                });
            })
            ->orderBy('created_at', 'desc');
    }

    public function getClassroomsProperty()
    {
        return Classroom::orderBy('name')->get();
    }

    public function render()
    {
        // Buscar referências ordenadas por mês/ano e estudante
        $allReferences = $this->getReferencesQuery()
            ->orderBy('fee_year', 'desc')
            ->orderBy('fee_month', 'desc')
            ->orderBy('student_id')
            ->orderBy('created_at', 'desc')
            ->get();

        // Agrupar por mês/ano
        $groupedReferences = $allReferences->groupBy(function($reference) {
            return $reference->fee_month . '-' . $reference->fee_year;
        });

        $stats = [
            'pending' => PaymentReference::where('approval_status', 'pending_approval')->count(),
            'approved_today' => PaymentReference::where('approval_status', 'approved')
                ->whereDate('approved_at', today())
                ->count(),
            'rejected_today' => PaymentReference::where('approval_status', 'rejected')
                ->whereDate('approved_at', today())
                ->count(),
        ];

        return view('livewire.admin.fee.approve-references', [
            'groupedReferences' => $groupedReferences,
            'stats' => $stats,
            'classrooms' => $this->classrooms,
            'isPca' => $this->isPca(),
        ]);
    }
}
