<?php

namespace App\Http\Livewire\Admin\Fee;

use App\Models\PaymentReference;
use App\Models\Classroom;
use Livewire\Component;
use Livewire\WithPagination;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\ReferencesReportExport;

class ReferencesReport extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    // Filtros
    public $filterClass = '';
    public $filterTurma = '';
    public $filterStatus = '';
    public $filterDateFrom = '';
    public $filterDateTo = '';
    public $filterMonth = '';
    public $filterYear = '';
    public $searchStudent = '';

    // Dados para dropdowns
    public $availableClasses = [];
    public $availableTurmas = [];
    public $availableYears = [];

    // Estatísticas
    public $stats = [];

    public function mount()
    {
        // Definir período padrão: mês atual
        $this->filterDateFrom = now()->startOfMonth()->format('Y-m-d');
        $this->filterDateTo = now()->endOfMonth()->format('Y-m-d');
        $this->filterYear = date('Y');

        $this->loadAvailableData();
        $this->loadStats();
    }

    public function loadAvailableData()
    {
        // Classes disponíveis (1-12)
        $this->availableClasses = range(1, 12);

        // Anos disponíveis (últimos 3 anos + próximos 2)
        $currentYear = date('Y');
        $this->availableYears = range($currentYear - 3, $currentYear + 2);
    }

    public function updatedFilterClass($class)
    {
        $this->filterTurma = '';

        if ($class) {
            $this->availableTurmas = Classroom::where('class', $class)
                ->orderBy('name')
                ->get();
        } else {
            $this->availableTurmas = [];
        }

        $this->resetPage();
        $this->loadStats();
    }

    public function updatedFilterTurma()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedFilterStatus()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedFilterDateFrom()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedFilterDateTo()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedFilterMonth()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedFilterYear()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedSearchStudent()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->resetPage();
        $this->loadStats();
        session()->flash('success', 'Filtros aplicados!');
    }

    public function resetFilters()
    {
        $this->reset([
            'filterClass',
            'filterTurma',
            'filterStatus',
            'filterMonth',
            'filterYear',
            'searchStudent'
        ]);

        $this->filterDateFrom = now()->startOfMonth()->format('Y-m-d');
        $this->filterDateTo = now()->endOfMonth()->format('Y-m-d');
        $this->filterYear = date('Y');

        $this->availableTurmas = [];
        $this->resetPage();
        $this->loadStats();

        session()->flash('info', 'Filtros limpos!');
    }

    private function getQuery()
    {
        $query = PaymentReference::with(['student', 'student.classroom']);

        // Filtro por classe
        if ($this->filterClass) {
            $query->where('classe', $this->filterClass);
        }

        // Filtro por turma
        if ($this->filterTurma) {
            $classroom = Classroom::find($this->filterTurma);
            if ($classroom) {
                $query->where('turma', $classroom->name);
            }
        }

        // Filtro por status
        if ($this->filterStatus) {
            $query->where('status', $this->filterStatus);
        }

        // Filtro por data de criação
        if ($this->filterDateFrom) {
            $query->whereDate('created_at', '>=', $this->filterDateFrom);
        }

        if ($this->filterDateTo) {
            $query->whereDate('created_at', '<=', $this->filterDateTo);
        }

        // Filtro por mês de pagamento
        if ($this->filterMonth) {
            $query->where('fee_month', $this->filterMonth);
        }

        // Filtro por ano de pagamento
        if ($this->filterYear) {
            $query->where('fee_year', $this->filterYear);
        }

        // Busca por estudante
        if ($this->searchStudent) {
            $query->whereHas('student', function($q) {
                $q->where('name', 'like', '%' . $this->searchStudent . '%')
                  ->orWhere('student_id', 'like', '%' . $this->searchStudent . '%');
            });
        }

        return $query;
    }

    public function loadStats()
    {
        $query = $this->getQuery();

        $this->stats = [
            'total_references' => $query->count(),
            'total_amount' => $query->sum('amount'),
            'pending' => $query->where('status', 'pending')->count(),
            'paid' => $query->where('status', 'paid')->count(),
            'expired' => $query->where('status', 'expired')->count(),
            'pending_amount' => $query->where('status', 'pending')->sum('amount'),
            'paid_amount' => $query->where('status', 'paid')->sum('amount'),
        ];
    }

    public function exportExcel()
    {
        $references = $this->getQuery()->get();

        $fileName = 'relatorio_referencias_' . now()->format('Y-m-d_His') . '.xlsx';

        return Excel::download(new ReferencesReportExport($references, $this->stats), $fileName);
    }

    public function render()
    {
        $references = $this->getQuery()
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('livewire.admin.fee.references-report', [
            'references' => $references,
        ]);
    }
}
