<?php

namespace App\Http\Livewire\Admin\Guardians;

use App\Models\Guardian;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Livewire\Component;
use Livewire\WithPagination;

class ListGuardians extends Component
{
    use WithPagination;

    public $search = '';
    public $showForm = false;
    public $editId = null;

    // Formulário
    public $name = '';
    public $phone = '';
    public $email = '';
    public $relationship = 'encarregado';
    public $is_active = true;

    // Ligação estudante
    public $linkStudentId = '';
    public $linkRelationship = 'encarregado';
    public $linkGuardianId = null;
    public $showLinkModal = false;

    // Reset password
    public $resetGuardianId = null;
    public $newPassword = '';
    public $showResetModal = false;

    // Credenciais
    public $credentialsGuardian = null;
    public $showCredentialsModal = false;

    protected $queryString = ['search'];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function getGuardiansProperty()
    {
        return Guardian::withTrashed(false)
            ->withCount('students')
            ->when($this->search, fn($q) =>
                $q->where('name', 'like', "%{$this->search}%")
                  ->orWhere('phone', 'like', "%{$this->search}%")
            )
            ->orderBy('name')
            ->paginate(20);
    }

    public function getStudentsForLinkProperty()
    {
        if (!$this->linkGuardianId) return collect([]);
        $guardian = Guardian::find($this->linkGuardianId);
        if (!$guardian) return collect([]);
        $linkedIds = $guardian->students()->pluck('users.id')->toArray();
        return User::whereNotIn('id', $linkedIds)->orderBy('name')->get(['id','name','student_id','classroom_id']);
    }

    // --- CRUD ---
    public function openCreate()
    {
        $this->reset(['name','phone','email','relationship','is_active','editId']);
        $this->relationship = 'encarregado';
        $this->is_active = true;
        $this->showForm = true;
    }

    public function openEdit(string $id)
    {
        $g = Guardian::findOrFail($id);
        $this->editId       = $g->id;
        $this->name         = $g->name;
        $this->phone        = $g->phone;
        $this->email        = $g->email ?? '';
        $this->relationship = $g->relationship;
        $this->is_active    = $g->is_active;
        $this->showForm     = true;
    }

    public function save()
    {
        $this->validate([
            'name'         => 'required|string|min:3',
            'phone'        => 'required|string|min:9',
            'relationship' => 'required|in:pai,mae,encarregado,outro',
        ]);

        $phone = preg_replace('/\D/', '', $this->phone);
        if (strlen($phone) === 9 && !str_starts_with($phone, '258')) {
            $phone = '258' . $phone;
        }

        if ($this->editId) {
            $g = Guardian::findOrFail($this->editId);
            $g->update([
                'name'         => $this->name,
                'phone'        => $phone,
                'email'        => $this->email ?: null,
                'relationship' => $this->relationship,
                'is_active'    => $this->is_active,
            ]);
            $this->emit('alert', ['type' => 'success', 'message' => 'Encarregado actualizado.']);
        } else {
            // Verificar se já existe
            if (Guardian::where('phone', $phone)->exists()) {
                $this->addError('phone', 'Já existe um encarregado com este número de telefone.');
                return;
            }
            Guardian::create([
                'name'         => $this->name,
                'phone'        => $phone,
                'email'        => $this->email ?: null,
                'relationship' => $this->relationship,
                'is_active'    => $this->is_active,
                'password'     => Hash::make(substr($phone, -6)), // últimos 6 dígitos
            ]);
            $this->emit('alert', ['type' => 'success', 'message' => 'Encarregado criado. Password padrão: últimos 6 dígitos do telefone.']);
        }

        $this->showForm = false;
        $this->reset(['name','phone','email','relationship','is_active','editId']);
    }

    public function toggleActive(string $id)
    {
        $g = Guardian::findOrFail($id);
        $g->update(['is_active' => !$g->is_active]);
    }

    // --- Link student ---
    public function openLinkModal(string $guardianId)
    {
        $this->linkGuardianId  = $guardianId;
        $this->linkStudentId   = '';
        $this->linkRelationship = 'encarregado';
        $this->showLinkModal   = true;
    }

    public function linkStudent()
    {
        $this->validate([
            'linkStudentId'   => 'required|exists:users,id',
            'linkRelationship' => 'required',
        ]);

        $g = Guardian::findOrFail($this->linkGuardianId);
        $g->students()->syncWithoutDetaching([
            $this->linkStudentId => ['relationship' => $this->linkRelationship, 'is_primary' => false]
        ]);

        $this->showLinkModal = false;
        $this->emit('alert', ['type' => 'success', 'message' => 'Estudante associado ao encarregado.']);
    }

    public function unlinkStudent(string $guardianId, string $studentId)
    {
        $g = Guardian::findOrFail($guardianId);
        $g->students()->detach($studentId);
        $this->emit('alert', ['type' => 'success', 'message' => 'Ligação removida.']);
    }

    // --- Credenciais ---
    public function openCredentials(string $guardianId)
    {
        $this->credentialsGuardian = Guardian::findOrFail($guardianId);
        $this->showCredentialsModal = true;
    }

    // --- Reset password ---
    public function openResetPassword(string $guardianId)
    {
        $this->resetGuardianId = $guardianId;
        $this->newPassword     = '';
        $this->showResetModal  = true;
    }

    public function resetPassword()
    {
        $this->validate(['newPassword' => 'required|min:6']);
        Guardian::findOrFail($this->resetGuardianId)->update(['password' => Hash::make($this->newPassword)]);
        $this->showResetModal = false;
        $this->emit('alert', ['type' => 'success', 'message' => 'Palavra-passe redefinida com sucesso.']);
    }

    public function render()
    {
        return view('livewire.admin.guardians.list-guardians', [
            'guardians'         => $this->guardians,
            'students_for_link' => $this->students_for_link,
        ]);
    }
}
