<?php

namespace App\Http\Livewire\Admin\POS;

use App\Models\POS\Category;
use App\Models\POS\Product;
use Illuminate\Support\Str;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;

class ManageProducts extends Component
{
    use LivewireAlert, WithPagination, WithFileUploads;

    public $search = '';
    public $categoryFilter = '';
    public $stockFilter = '';

    // Form fields
    public $productId;
    public $name, $category_id, $sku, $description;
    public $cost_price, $selling_price, $discount_price;
    public $quantity, $min_stock = 5, $unit = 'Unidade';
    public $is_active = true, $track_stock = true;
    public $image;
    public $existingImage;

    public $showModal = false;
    public $editMode = false;

    protected $listeners = [
        'confirmedDelete',
        'cancelledDelete'
    ];

    public function rules()
    {
        $rules = [
            'name' => 'required|string|max:255',
            'category_id' => 'required|uuid|exists:pos_categories,id',
            'sku' => 'required|string|max:255|unique:pos_products,sku,' . ($this->productId ?? 'NULL') . ',id',
            'cost_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
            'discount_price' => 'nullable|numeric|min:0',
            'quantity' => 'required|integer|min:0',
            'min_stock' => 'required|integer|min:0',
            'unit' => 'required|string|max:50',
            'is_active' => 'boolean',
            'track_stock' => 'boolean',
            'description' => 'nullable|string',
        ];

        if (!$this->editMode || $this->image) {
            $rules['image'] = 'nullable|image|max:2048';
        }

        return $rules;
    }

    public function openModal()
    {
        $this->resetForm();
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    public function edit($id)
    {
        $product = Product::findOrFail($id);

        $this->productId = $product->id;
        $this->name = $product->name;
        $this->category_id = $product->category_id;
        $this->sku = $product->sku;
        $this->description = $product->description;
        $this->cost_price = $product->cost_price;
        $this->selling_price = $product->selling_price;
        $this->discount_price = $product->discount_price;
        $this->quantity = $product->quantity;
        $this->min_stock = $product->min_stock;
        $this->unit = $product->unit;
        $this->is_active = $product->is_active;
        $this->track_stock = $product->track_stock;
        $this->existingImage = $product->image;

        $this->editMode = true;
        $this->showModal = true;
    }

    public function save()
    {
        $this->validate();

        try {
            $data = [
                'name' => $this->name,
                'category_id' => $this->category_id,
                'sku' => $this->sku,
                'description' => $this->description,
                'cost_price' => $this->cost_price,
                'selling_price' => $this->selling_price,
                'discount_price' => $this->discount_price,
                'quantity' => $this->quantity,
                'min_stock' => $this->min_stock,
                'unit' => $this->unit,
                'is_active' => $this->is_active,
                'track_stock' => $this->track_stock,
            ];

            if ($this->image) {
                $data['image'] = $this->image->store('pos/products', 'public');
            }

            if ($this->editMode) {
                $product = Product::findOrFail($this->productId);
                $product->update($data);
                $message = 'Produto atualizado com sucesso';
            } else {
                Product::create($data);
                $message = 'Produto criado com sucesso';
            }

            $this->alert('success', $message);
            $this->closeModal();
            $this->resetPage();
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao salvar produto: ' . $e->getMessage());
        }
    }

    public function confirmDelete($id)
    {
        $this->productId = $id;
        $this->confirm('Tem certeza que deseja excluir este produto?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, excluir',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedDelete',
            'onDismissed' => 'cancelledDelete'
        ]);
    }

    public function confirmedDelete()
    {
        try {
            Product::findOrFail($this->productId)->delete();
            $this->alert('success', 'Produto excluído com sucesso');
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao excluir produto: ' . $e->getMessage());
        }
    }

    public function cancelledDelete()
    {
        $this->alert('info', 'Exclusão cancelada');
    }

    public function resetForm()
    {
        $this->reset([
            'productId', 'name', 'category_id', 'sku', 'description',
            'cost_price', 'selling_price', 'discount_price',
            'quantity', 'min_stock', 'unit', 'image', 'existingImage'
        ]);
        $this->is_active = true;
        $this->track_stock = true;
        $this->min_stock = 5;
        $this->unit = 'Unidade';
        $this->editMode = false;
    }

    public function render()
    {
        $query = Product::with('category')
            ->when($this->search, function($q) {
                $q->where(function($query) {
                    $query->where('name', 'like', '%' . $this->search . '%')
                          ->orWhere('sku', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->categoryFilter, function($q) {
                $q->where('category_id', $this->categoryFilter);
            })
            ->when($this->stockFilter, function($q) {
                if ($this->stockFilter == 'low') {
                    $q->lowStock();
                } elseif ($this->stockFilter == 'out') {
                    $q->outOfStock();
                }
            })
            ->orderBy('created_at', 'desc');

        return view('livewire.admin.pos.manage-products', [
            'products' => $query->paginate(20),
            'categories' => Category::active()->get(),
        ]);
    }
}
