<?php

namespace App\Http\Livewire\Admin\Permissions;

use App\Models\Admin;
use App\Models\AdminPermission;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;

class ManagePermissions extends Component
{
    use LivewireAlert;

    public $admins;
    public $selectedAdmin;
    public $adminPermissions = [];
    public $availablePermissions = [];
    public $search = '';

    protected $listeners = [
        'confirmedSave',
        'cancelledSave',
        'confirmedRoleChange',
        'cancelledRoleChange',
        'confirmedChangeRole'
    ];

    public function mount()
    {
        $this->loadAdmins();
        $this->availablePermissions = AdminPermission::availablePermissions();
    }

    public function loadAdmins()
    {
        $query = Admin::where('is_active', 1);

        if ($this->search) {
            $query->where(function($q) {
                $q->where('name', 'like', '%' . $this->search . '%')
                  ->orWhere('email', 'like', '%' . $this->search . '%');
            });
        }

        $this->admins = $query->orderBy('name')->get();
    }

    public function updatedSearch()
    {
        $this->loadAdmins();
    }

    public function selectAdmin($adminId)
    {
        $this->selectedAdmin = Admin::find($adminId);
        $this->loadAdminPermissions();
    }

    public function loadAdminPermissions()
    {
        if (!$this->selectedAdmin) {
            $this->adminPermissions = [];
            return;
        }

        // Get all permissions (role defaults + custom)
        $this->adminPermissions = $this->selectedAdmin->getPermissions();
    }

    public function togglePermission($permissionKey)
    {
        if (!$this->selectedAdmin) {
            $this->alert('warning', 'Selecione um usuário primeiro');
            return;
        }

        // Check if permission is currently granted
        $index = array_search($permissionKey, $this->adminPermissions);

        if ($index !== false) {
            // Remove permission
            unset($this->adminPermissions[$index]);
            $this->adminPermissions = array_values($this->adminPermissions);
        } else {
            // Add permission
            $this->adminPermissions[] = $permissionKey;
        }
    }

    public function hasPermission($permissionKey)
    {
        return in_array($permissionKey, $this->adminPermissions);
    }

    public function save()
    {
        if (!$this->selectedAdmin) {
            $this->alert('warning', 'Selecione um usuário primeiro');
            return;
        }

        $this->confirm('Deseja salvar as permissões?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, salvar',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedSave',
            'onDismissed' => 'cancelledSave'
        ]);
    }

    public function confirmedSave()
    {
        try {
            // Get role default permissions
            $roleDefaults = $this->selectedAdmin->getRoleDefaultPermissions();

            // Delete all existing custom permissions for this admin
            AdminPermission::where('admin_id', $this->selectedAdmin->id)->delete();

            // Save only custom permissions (extras or revocations)
            foreach ($this->adminPermissions as $permissionKey) {
                // If not in role defaults, it's a custom grant
                if (!in_array($permissionKey, $roleDefaults)) {
                    AdminPermission::create([
                        'admin_id' => $this->selectedAdmin->id,
                        'permission_key' => $permissionKey,
                        'granted' => true,
                    ]);
                }
            }

            // Check for revocations (role defaults that are unchecked)
            foreach ($roleDefaults as $defaultPerm) {
                if (!in_array($defaultPerm, $this->adminPermissions)) {
                    AdminPermission::create([
                        'admin_id' => $this->selectedAdmin->id,
                        'permission_key' => $defaultPerm,
                        'granted' => false, // Explicitly revoke
                    ]);
                }
            }

            $this->alert('success', 'Permissões salvas com sucesso!');
            $this->loadAdminPermissions();

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao salvar permissões: ' . $e->getMessage());
        }
    }

    public function cancelledSave()
    {
        $this->alert('info', 'Operação cancelada');
    }

    public function togglePcaStatus()
    {
        if (!$this->selectedAdmin) {
            $this->alert('warning', 'Selecione um usuário primeiro');
            return;
        }

        $currentStatus = $this->selectedAdmin->role == 0;
        $newStatus = !$currentStatus;

        $message = $newStatus
            ? 'Deseja tornar este usuário um PCA (Super Admin)? Ele terá acesso total ao sistema.'
            : 'Deseja remover o status de PCA deste usuário? Ele perderá acesso total ao sistema.';

        $this->confirm($message, [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, alterar',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedRoleChange',
            'onDismissed' => 'cancelledRoleChange'
        ]);
    }

    public function confirmedRoleChange()
    {
        try {
            $currentStatus = $this->selectedAdmin->role == 0;

            if ($currentStatus) {
                // Remover PCA - definir como Reg Academico (role 1)
                $this->selectedAdmin->role = 1;
                $successMessage = 'Status de PCA removido. Usuário agora é Reg Academico.';
            } else {
                // Tornar PCA
                $this->selectedAdmin->role = 0;
                $successMessage = 'Usuário promovido a PCA (Super Admin) com sucesso!';
            }

            $this->selectedAdmin->save();

            // Deletar todas as permissões customizadas ao mudar de role
            // Isso garante que o usuário tenha apenas as permissões padrão do novo role
            AdminPermission::where('admin_id', $this->selectedAdmin->id)->delete();

            // Recarregar admin e permissões
            $this->selectedAdmin = Admin::find($this->selectedAdmin->id);
            $this->loadAdminPermissions();
            $this->loadAdmins();

            $this->alert('success', $successMessage . ' As permissões foram resetadas para as padrões do novo role.');

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao alterar status: ' . $e->getMessage());
        }
    }

    public function cancelledRoleChange()
    {
        $this->alert('info', 'Operação cancelada');
    }

    public function changeRole($newRole)
    {
        if (!$this->selectedAdmin) {
            $this->alert('warning', 'Selecione um usuário primeiro');
            return;
        }

        // Converter para inteiro
        $newRole = (int) $newRole;

        if ($newRole == $this->selectedAdmin->role) {
            return; // Sem mudança
        }

        $roleNames = [
            0 => 'PCA (Super Admin)',
            1 => 'Reg Academico',
            2 => 'Professor',
            3 => 'Diretor',
            4 => 'Admin',
            5 => 'Contabilidade',
            6 => 'Secretaria'
        ];

        // Validar se o role é válido
        if (!array_key_exists($newRole, $roleNames)) {
            $this->alert('error', 'Role inválido');
            return;
        }

        $this->confirm("Deseja alterar o role deste usuário para '{$roleNames[$newRole]}'?", [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, alterar',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedChangeRole',
            'onDismissed' => 'cancelledRoleChange',
            'data' => ['newRole' => (int) $newRole]
        ]);
    }

    public function confirmedChangeRole($data = [])
    {
        try {
            // Tentar obter newRole de diferentes formas
            $newRole = null;

            if (is_array($data) && isset($data['newRole'])) {
                $newRole = (int) $data['newRole'];
            } elseif (is_array($data) && isset($data['data']['newRole'])) {
                $newRole = (int) $data['data']['newRole'];
            }

            if ($newRole === null || $newRole === 0 && !isset($data['newRole'])) {
                $this->alert('error', 'Role inválido. Dados recebidos: ' . json_encode($data));
                return;
            }

            $roleNames = [
                0 => 'PCA (Super Admin)',
                1 => 'Reg Academico',
                2 => 'Professor',
                3 => 'Diretor',
                4 => 'Admin',
                5 => 'Contabilidade',
                6 => 'Secretaria'
            ];

            // Validar role
            if (!array_key_exists($newRole, $roleNames)) {
                $this->alert('error', 'Role inválido: ' . $newRole);
                return;
            }

            // Salvar o role antigo para comparação
            $oldRole = $this->selectedAdmin->role;

            $this->selectedAdmin->role = $newRole;
            $this->selectedAdmin->save();

            // Deletar todas as permissões customizadas ao mudar de role
            // Isso garante que o usuário tenha apenas as permissões padrão do novo role
            AdminPermission::where('admin_id', $this->selectedAdmin->id)->delete();

            // Recarregar admin e permissões
            $this->selectedAdmin = Admin::find($this->selectedAdmin->id);
            $this->loadAdminPermissions();
            $this->loadAdmins();

            $this->alert('success', "Role alterado para '{$roleNames[$newRole]}' com sucesso! As permissões foram atualizadas para as padrões do novo role.");

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao alterar role: ' . $e->getMessage());
        }
    }

    public function grantAll()
    {
        if (!$this->selectedAdmin) {
            $this->alert('warning', 'Selecione um usuário primeiro');
            return;
        }

        $this->adminPermissions = [];

        foreach ($this->availablePermissions as $group) {
            foreach ($group['permissions'] as $key => $label) {
                $this->adminPermissions[] = $key;
            }
        }

        $this->alert('info', 'Todas as permissões foram marcadas. Clique em Salvar para confirmar.');
    }

    public function revokeAll()
    {
        if (!$this->selectedAdmin) {
            $this->alert('warning', 'Selecione um usuário primeiro');
            return;
        }

        $this->adminPermissions = [];
        $this->alert('info', 'Todas as permissões foram desmarcadas. Clique em Salvar para confirmar.');
    }

    public function applyQuickGroup($groupName)
    {
        if (!$this->selectedAdmin) {
            $this->alert('warning', 'Selecione um usuário primeiro');
            return;
        }

        $quickGroups = $this->getQuickGroups();

        if (isset($quickGroups[$groupName])) {
            $this->adminPermissions = $quickGroups[$groupName]['permissions'];
            $this->alert('success', "Grupo '{$quickGroups[$groupName]['label']}' aplicado! Clique em Salvar para confirmar.");
        }
    }

    public function getQuickGroups()
    {
        return [
            'pos_full' => [
                'label' => 'POS Completo',
                'icon' => 'cash-register',
                'color' => 'primary',
                'permissions' => [
                    'pos.access', 'pos.dashboard', 'pos.sales', 'pos.sales_report',
                    'pos.products.view', 'pos.products.manage',
                    'pos.categories.view', 'pos.categories.manage',
                    'pos.settings',
                ],
            ],
            'pos_cashier' => [
                'label' => 'Caixa POS',
                'icon' => 'money-bill-wave',
                'color' => 'success',
                'permissions' => [
                    'pos.access', 'pos.sales',
                    'pos.products.view', 'pos.categories.view',
                ],
            ],
            'academic_full' => [
                'label' => 'Acadêmico Completo',
                'icon' => 'graduation-cap',
                'color' => 'info',
                'permissions' => [
                    'students.view', 'students.add', 'students.edit', 'students.export',
                    'teachers.view', 'teachers.add', 'teachers.edit', 'teachers.assign',
                    'classes.view', 'classes.add', 'classes.edit',
                    'exams.view', 'exams.add', 'exams.edit', 'exams.reports',
                ],
            ],
            'teacher' => [
                'label' => 'Professor',
                'icon' => 'chalkboard-teacher',
                'color' => 'warning',
                'permissions' => [
                    'students.view',
                    'classes.view',
                    'exams.view', 'exams.add', 'exams.edit', 'exams.reports',
                    'library.view',
                ],
            ],
            'finance' => [
                'label' => 'Financeiro',
                'icon' => 'dollar-sign',
                'color' => 'success',
                'permissions' => [
                    'payments.view', 'payments.collect', 'payments.manage', 'payments.reports',
                    'pos.access', 'pos.sales', 'pos.sales_report',
                ],
            ],
            'librarian' => [
                'label' => 'Bibliotecário',
                'icon' => 'book',
                'color' => 'secondary',
                'permissions' => [
                    'library.view', 'library.add', 'library.edit', 'library.delete',
                ],
            ],
        ];
    }

    public function render()
    {
        // Obter permissões padrão do role atual para comparação
        $roleDefaults = [];
        if ($this->selectedAdmin) {
            $roleDefaults = $this->selectedAdmin->getRoleDefaultPermissions();
        }

        return view('livewire.admin.permissions.manage-permissions', [
            'quickGroups' => $this->getQuickGroups(),
            'roleDefaults' => $roleDefaults,
        ]);
    }
}
