<?php

namespace App\Http\Livewire\Admin\Secretary;

use App\Models\Classroom;
use App\Models\EnrollmentRenewal;
use App\Models\PreEnrollment;
use App\Models\SystemSetting;
use Livewire\Component;

class EnrollmentDashboard extends Component
{
    public string $year = '';

    public function mount(): void
    {
        $this->year = (string) SystemSetting::get('renewal_academic_year', (string) now()->year, 'enrollment');
    }

    public function render()
    {
        $year = $this->year ?: (string) now()->year;

        // ── Pré-inscrições (novos alunos externos) ────────────────────────
        $preBase = PreEnrollment::where('academic_year', $year);

        $preTotal      = (clone $preBase)->count();
        $prePendente   = (clone $preBase)->where('status', 'pendente')->count();
        $preConfirmada = (clone $preBase)->where('status', 'confirmada')->count();
        $preMatriculada= (clone $preBase)->where('status', 'matriculada')->count();
        $preAnulada    = (clone $preBase)->where('status', 'anulada')->count();

        $preBySex = (clone $preBase)
            ->selectRaw('candidate_sex, count(*) as total')
            ->groupBy('candidate_sex')
            ->pluck('total', 'candidate_sex')
            ->toArray();

        $preByClass = (clone $preBase)
            ->selectRaw('desired_class, status, count(*) as total')
            ->groupBy('desired_class', 'status')
            ->orderByRaw('CAST(desired_class AS UNSIGNED)')
            ->get()
            ->groupBy('desired_class');

        // ── Renovações (alunos internos) ──────────────────────────────────
        $renewBase = EnrollmentRenewal::where('academic_year', $year);

        $renewTotal        = (clone $renewBase)->count();
        $renewPendente     = (clone $renewBase)->where('status', 'pendente')->count();
        $renewConfirmada   = (clone $renewBase)->where('status', 'confirmada')->count();
        $renewCancelada    = (clone $renewBase)->where('status', 'cancelada')->count();
        $renewTransitioned = (clone $renewBase)->whereNotNull('transitioned_at')->count();

        // ── Vagas por turma ───────────────────────────────────────────────
        $classroomStats = Classroom::orderByRaw('CAST(class AS UNSIGNED)')
            ->orderBy('name')
            ->get()
            ->map(function ($classroom) use ($year) {
                $renewConfirmed = EnrollmentRenewal::where('from_classroom_id', $classroom->id)
                    ->where('academic_year', $year)
                    ->where('status', 'confirmada')
                    ->count();

                $newEnrolled = PreEnrollment::where('desired_classroom_id', $classroom->id)
                    ->where('academic_year', $year)
                    ->where('status', 'matriculada')
                    ->count();

                $used  = $renewConfirmed + $newEnrolled;
                $free  = $classroom->capacity ? max(0, $classroom->capacity - $used) : null;

                return [
                    'name'           => $classroom->name,
                    'class'          => $classroom->class,
                    'capacity'       => $classroom->capacity,
                    'renew_confirmed'=> $renewConfirmed,
                    'new_enrolled'   => $newEnrolled,
                    'used'           => $used,
                    'free'           => $free,
                ];
            })
            ->filter(fn($c) => $c['used'] > 0 || $c['capacity']);

        // ── Anos disponíveis para dropdown ────────────────────────────────
        $availableYears = PreEnrollment::selectRaw('academic_year')->distinct()->pluck('academic_year')
            ->merge(EnrollmentRenewal::selectRaw('academic_year')->distinct()->pluck('academic_year'))
            ->push((string) now()->year)
            ->push($year)
            ->filter()
            ->unique()
            ->sort()
            ->values();

        return view('livewire.admin.secretary.enrollment-dashboard', compact(
            'year', 'availableYears',
            'preTotal', 'prePendente', 'preConfirmada', 'preMatriculada', 'preAnulada',
            'preBySex', 'preByClass',
            'renewTotal', 'renewPendente', 'renewConfirmada', 'renewCancelada', 'renewTransitioned',
            'classroomStats'
        ));
    }
}
