<?php

namespace App\Http\Livewire\Admin\Secretary;

use App\Models\Classroom;
use App\Models\PreEnrollment;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithPagination;

class ListPreEnrollments extends Component
{
    use WithPagination, FindGuard, LivewireAlert;

    protected string $paginationTheme = 'bootstrap';

    public $search;
    public $selectedStatus;
    public $selectedYear;
    public $selectedClass;
    public bool $readyToLoad = false;

    protected $queryString = ['search', 'selectedStatus', 'selectedYear', 'selectedClass'];

    protected $listeners = ['confirmed', 'cancelled'];

    public $actionId;
    public $actionType;

    public function mount()
    {
        $this->selectedYear = date('Y');
    }

    public function loadData()
    {
        $this->readyToLoad = true;
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingSelectedStatus()
    {
        $this->resetPage();
    }

    public function anularPreEnrollment($id)
    {
        $this->actionId = $id;
        $this->actionType = 'anular';
        $this->confirm('Tem certeza de que deseja anular esta pré-inscrição?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, anular!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        if ($this->actionId && $this->actionType === 'anular') {
            $pre = PreEnrollment::find($this->actionId);
            if ($pre && $pre->isPendente()) {
                $pre->update([
                    'status' => 'anulada',
                    'cancelled_at' => now(),
                ]);
                $this->alert('success', 'Pré-inscrição anulada.');
            }
        }
        $this->actionId = null;
        $this->actionType = null;
    }

    public function cancelled()
    {
        $this->alert('info', 'Operação cancelada.');
        $this->actionId = null;
        $this->actionType = null;
    }

    public function render()
    {
        $this->findGuardType()->user();

        $records = [];
        if ($this->readyToLoad) {
            $query = PreEnrollment::query()
                ->with(['desiredClassroom'])
                ->when($this->selectedYear, fn($q) => $q->where('academic_year', $this->selectedYear))
                ->when($this->selectedStatus, fn($q) => $q->where('status', $this->selectedStatus))
                ->when($this->selectedClass, fn($q) => $q->where('desired_class', $this->selectedClass))
                ->when($this->search, function ($q) {
                    $q->where(function ($sub) {
                        $sub->where('candidate_name', 'like', '%' . $this->search . '%')
                            ->orWhere('code', 'like', '%' . $this->search . '%')
                            ->orWhere('candidate_phone', 'like', '%' . $this->search . '%');
                    });
                })
                ->latest('submitted_at');

            $records = $query->paginate(25);
        }

        // Stats
        $stats = [];
        if ($this->readyToLoad) {
            $baseQuery = PreEnrollment::query()
                ->when($this->selectedYear, fn($q) => $q->where('academic_year', $this->selectedYear));
            $stats = [
                'total' => (clone $baseQuery)->count(),
                'pendentes' => (clone $baseQuery)->where('status', 'pendente')->count(),
                'confirmadas' => (clone $baseQuery)->where('status', 'confirmada')->count(),
                'matriculadas' => (clone $baseQuery)->where('status', 'matriculada')->count(),
                'anuladas' => (clone $baseQuery)->where('status', 'anulada')->count(),
            ];
        }

        $availableClasses = Classroom::where('is_active', true)
            ->select('class')
            ->distinct()
            ->orderByRaw('CAST(class AS UNSIGNED)')
            ->pluck('class');

        return view('livewire.admin.secretary.list-pre-enrollments', [
            'records' => $records,
            'stats' => $stats,
            'availableClasses' => $availableClasses,
        ]);
    }
}
