<?php

namespace App\Http\Livewire\Admin\Settings;

use App\Models\SystemSetting;
use App\Services\SmsService;
use Illuminate\Support\Facades\Auth;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;

class SmsSettings extends Component
{
    use LivewireAlert;

    // Gateway selection
    public $gateway = 'movitel';

    // Movitel settings
    public $movitel_api_url = 'https://sms.itcore.app/api/http/sms/send';
    public $movitel_api_token = '';
    public $movitel_sender_id = 'COPMOZ';

    // Twilio settings
    public $twilio_sid = '';
    public $twilio_token = '';
    public $twilio_from = 'COPMOZ';

    // General settings
    public $country_code = '258';
    public $log_messages = true;

    // Test SMS
    public $test_phone = '';
    public $test_message = 'Mensagem de teste do sistema COPMOZ';

    protected $listeners = [
        'confirmedSave',
        'cancelledSave'
    ];

    protected function rules()
    {
        return [
            'gateway' => 'required|in:movitel,twilio',
            'movitel_api_url' => 'required_if:gateway,movitel|url',
            'movitel_api_token' => 'required_if:gateway,movitel',
            'movitel_sender_id' => 'required_if:gateway,movitel|max:11',
            'twilio_sid' => 'required_if:gateway,twilio',
            'twilio_token' => 'required_if:gateway,twilio',
            'twilio_from' => 'required_if:gateway,twilio',
            'country_code' => 'required|numeric|digits_between:1,4',
            'test_phone' => 'nullable|numeric|digits_between:9,15',
        ];
    }

    protected $messages = [
        'movitel_api_token.required_if' => 'O token da API Movitel é obrigatório.',
        'twilio_sid.required_if' => 'O SID do Twilio é obrigatório.',
        'twilio_token.required_if' => 'O Token do Twilio é obrigatório.',
    ];

    public function mount()
    {
        // Only Super Admin (role 0) or users with sms.settings permission can access
        $user = Auth::guard('admin')->user();

        if (!$user || ($user->role != 0 && !$user->hasPermission('sms.settings'))) {
            session()->flash('error', 'Acesso negado. Apenas administradores podem acessar as configurações de SMS.');
            return redirect()->route('admin.home');
        }

        $this->loadSettings();
    }

    public function loadSettings()
    {
        $this->gateway = SystemSetting::get('gateway', 'movitel', 'sms');
        $this->movitel_api_url = SystemSetting::get('movitel_api_url', 'https://sms.itcore.app/api/http/sms/send', 'sms');
        $this->movitel_api_token = SystemSetting::get('movitel_api_token', '', 'sms');
        $this->movitel_sender_id = SystemSetting::get('movitel_sender_id', 'COPMOZ', 'sms');
        $this->twilio_sid = SystemSetting::get('twilio_sid', '', 'sms');
        $this->twilio_token = SystemSetting::get('twilio_token', '', 'sms');
        $this->twilio_from = SystemSetting::get('twilio_from', 'COPMOZ', 'sms');
        $this->country_code = SystemSetting::get('country_code', '258', 'sms');
        $this->log_messages = SystemSetting::get('log_messages', true, 'sms');
    }

    public function save()
    {
        $this->validate();

        $this->confirm('Deseja salvar as configurações de SMS?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, salvar',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedSave',
            'onDismissed' => 'cancelledSave'
        ]);
    }

    public function confirmedSave()
    {
        try {
            // Limpar cache ANTES de salvar para garantir que não há valores antigos
            SystemSetting::clearGroupCache('sms');
            \Illuminate\Support\Facades\Cache::flush();

            SystemSetting::set('gateway', $this->gateway, 'string', 'sms');
            SystemSetting::set('movitel_api_url', $this->movitel_api_url, 'string', 'sms');
            SystemSetting::set('movitel_api_token', $this->movitel_api_token, 'string', 'sms');
            SystemSetting::set('movitel_sender_id', $this->movitel_sender_id, 'string', 'sms');
            SystemSetting::set('twilio_sid', $this->twilio_sid, 'string', 'sms');
            SystemSetting::set('twilio_token', $this->twilio_token, 'string', 'sms');
            SystemSetting::set('twilio_from', $this->twilio_from, 'string', 'sms');
            SystemSetting::set('country_code', $this->country_code, 'string', 'sms');
            SystemSetting::set('log_messages', $this->log_messages ? 'true' : 'false', 'boolean', 'sms');

            // Limpar cache APÓS salvar
            SystemSetting::clearGroupCache('sms');

            $this->alert('success', 'Configurações de SMS salvas com sucesso!');
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao salvar configurações: ' . $e->getMessage());
        }
    }

    public function cancelledSave()
    {
        $this->alert('info', 'Operação cancelada');
    }

    public function sendTestSms()
    {
        if (empty($this->test_phone)) {
            $this->alert('error', 'Insira um número de telefone para teste');
            return;
        }

        // Validar token antes de enviar
        if ($this->gateway === 'movitel' && empty($this->movitel_api_token)) {
            $this->alert('error', 'Insira o API Token do Movitel antes de testar');
            return;
        }

        if ($this->gateway === 'twilio' && (empty($this->twilio_sid) || empty($this->twilio_token))) {
            $this->alert('error', 'Insira as credenciais do Twilio antes de testar');
            return;
        }

        try {
            // Salvar configurações antes do teste
            $this->confirmedSave();

            // Log para debug
            \Log::info("SMS Test - Gateway: {$this->gateway}, Token length: " . strlen($this->movitel_api_token));

            // Criar service e configurar directamente com os valores do formulário
            $smsService = new SmsService($this->gateway);

            if ($this->gateway === 'movitel') {
                $smsService->setConfig([
                    'api_url' => $this->movitel_api_url,
                    'api_token' => $this->movitel_api_token,
                    'sender_id' => $this->movitel_sender_id,
                ]);
            } else {
                $smsService->setConfig([
                    'sid' => $this->twilio_sid,
                    'token' => $this->twilio_token,
                    'from' => $this->twilio_from,
                ]);
            }

            // Log config para debug
            \Log::info("SMS Config: " . json_encode($smsService->getConfig()));

            $result = $smsService->send($this->test_phone, $this->test_message);

            if ($result['success']) {
                $this->alert('success', 'SMS de teste enviado com sucesso!');
            } else {
                $this->alert('error', 'Falha ao enviar SMS: ' . ($result['error'] ?? 'Erro desconhecido'));
            }
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao enviar SMS de teste: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.settings.sms-settings');
    }
}
