<?php

namespace App\Http\Livewire\Auth;

use App\Models\Guardian;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Livewire\Component;

class GuardianLogin extends Component
{
    public $phone = '';
    public $password = '';
    public $remember = false;

    protected $rules = [
        'phone'    => 'required|string',
        'password' => 'required|string|min:4',
    ];

    protected $messages = [
        'phone.required'    => 'O número de telefone é obrigatório.',
        'password.required' => 'A palavra-passe é obrigatória.',
        'password.min'      => 'A palavra-passe deve ter pelo menos 4 caracteres.',
    ];

    public function updated($field)
    {
        $this->validateOnly($field);
    }

    public function login()
    {
        $this->validate();

        // Normalizar telefone
        $phone = preg_replace('/\D/', '', $this->phone);
        if (strlen($phone) === 9 && !str_starts_with($phone, '258')) {
            $phone = '258' . $phone;
        }

        $guardian = Guardian::where('phone', $phone)->where('is_active', true)->first();

        if (!$guardian || !Hash::check($this->password, $guardian->password)) {
            $this->addError('phone', 'Credenciais inválidas. Verifique o telefone e a palavra-passe.');
            return;
        }

        Auth::guard('guardian')->login($guardian, $this->remember);

        $guardian->update(['last_login_at' => now()]);

        return redirect()->route('guardian.dashboard');
    }

    public function render()
    {
        return view('livewire.auth.guardian-login')
            ->layout('layouts.guardian');
    }
}
