<?php

namespace App\Http\Livewire\User\Attendance;

use App\Models\Attendance;
use Carbon\Carbon;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;

class StudentAttendance extends Component
{
    use FindGuard;

    public $year;
    public $month;
    public $week_start;
    public $selectedDate;
    public $viewMode = 'month'; // day, week, month, year
    public bool $readyToLoad = false;

    public function mount()
    {
        $this->year = (int) date('Y');
        $this->month = (int) date('m');
        $this->selectedDate = date('Y-m-d');
        $this->week_start = Carbon::now()->startOfWeek()->format('Y-m-d');
    }

    public function loadData()
    {
        $this->readyToLoad = true;
    }

    public function setViewMode($mode)
    {
        $this->viewMode = $mode;
    }

    public function selectDate($date)
    {
        $this->selectedDate = $date;
        $this->viewMode = 'day';
    }

    public function previousMonth()
    {
        $dt = Carbon::createFromDate($this->year, $this->month, 1)->subMonth();
        $this->year = $dt->year;
        $this->month = $dt->month;
    }

    public function nextMonth()
    {
        $dt = Carbon::createFromDate($this->year, $this->month, 1)->addMonth();
        $this->year = $dt->year;
        $this->month = $dt->month;
    }

    public function previousWeek()
    {
        $this->week_start = Carbon::parse($this->week_start)->subWeek()->format('Y-m-d');
    }

    public function nextWeek()
    {
        $this->week_start = Carbon::parse($this->week_start)->addWeek()->format('Y-m-d');
    }

    public function previousYear()
    {
        $this->year--;
    }

    public function nextYear()
    {
        $this->year++;
    }

    public function render()
    {
        $user = $this->findGuardType()->user();

        $summary = ['total' => 0, 'present' => 0, 'absent' => 0, 'late' => 0, 'justified' => 0, 'rate' => 0];
        $records = collect();
        $calendarData = [];
        $weekDays = [];
        $monthsSummary = [];

        if ($this->readyToLoad) {
            if ($this->viewMode === 'day') {
                // Registos do dia seleccionado
                $records = Attendance::where('student_id', $user->id)
                    ->where('date', $this->selectedDate)
                    ->orderBy('created_at')
                    ->get();

                $summary = $this->calculateSummary($records);

            } elseif ($this->viewMode === 'week') {
                $weekStart = Carbon::parse($this->week_start);
                $weekEnd = $weekStart->copy()->endOfWeek();

                $allRecords = Attendance::where('student_id', $user->id)
                    ->whereBetween('date', [$weekStart->format('Y-m-d'), $weekEnd->format('Y-m-d')])
                    ->orderBy('date')
                    ->orderBy('created_at')
                    ->get();

                $summary = $this->calculateSummary($allRecords);

                // Agrupar por dia da semana
                for ($i = 0; $i < 5; $i++) { // Seg-Sex
                    $day = $weekStart->copy()->addDays($i);
                    $dayRecords = $allRecords->where('date', $day->format('Y-m-d'));
                    $weekDays[] = [
                        'date' => $day->format('Y-m-d'),
                        'day_name' => $day->translatedFormat('l'),
                        'day' => $day->format('d'),
                        'month' => $day->format('m'),
                        'records' => $dayRecords,
                        'status' => $this->getDayStatus($dayRecords),
                    ];
                }
                $records = $allRecords;

            } elseif ($this->viewMode === 'month') {
                $allRecords = Attendance::where('student_id', $user->id)
                    ->whereMonth('date', $this->month)
                    ->whereYear('date', $this->year)
                    ->orderBy('date', 'desc')
                    ->orderBy('created_at')
                    ->get();

                $summary = $this->calculateSummary($allRecords);
                $records = $allRecords;

                // Dados do calendário
                $calendarData = $this->buildCalendarData($user->id);

            } elseif ($this->viewMode === 'year') {
                $allRecords = Attendance::where('student_id', $user->id)
                    ->whereYear('date', $this->year)
                    ->get();

                $summary = $this->calculateSummary($allRecords);

                // Resumo por mês
                for ($m = 1; $m <= 12; $m++) {
                    $monthRecords = $allRecords->filter(function ($r) use ($m) {
                        return $r->date->month == $m;
                    });
                    $monthsSummary[] = [
                        'month' => $m,
                        'month_name' => Carbon::createFromDate($this->year, $m, 1)->translatedFormat('F'),
                        'total' => $monthRecords->count(),
                        'present' => $monthRecords->where('status', 'present')->count(),
                        'absent' => $monthRecords->where('status', 'absent')->count(),
                        'late' => $monthRecords->where('status', 'late')->count(),
                        'justified' => $monthRecords->where('status', 'justified')->count(),
                        'rate' => $monthRecords->count() > 0
                            ? round(($monthRecords->where('status', 'present')->count() + $monthRecords->where('status', 'late')->count() + $monthRecords->where('status', 'justified')->count()) / $monthRecords->count() * 100, 1)
                            : 0,
                    ];
                }
            }
        }

        $monthNames = ['Janeiro','Fevereiro','Março','Abril','Maio','Junho','Julho','Agosto','Setembro','Outubro','Novembro','Dezembro'];

        return view('livewire.user.attendance.student-attendance', [
            'records' => $records,
            'summary' => $summary,
            'calendarData' => $calendarData,
            'weekDays' => $weekDays,
            'monthsSummary' => $monthsSummary,
            'monthNames' => $monthNames,
        ]);
    }

    private function calculateSummary($records)
    {
        $total = $records->count();
        $present = $records->where('status', 'present')->count();
        $absent = $records->where('status', 'absent')->count();
        $late = $records->where('status', 'late')->count();
        $justified = $records->where('status', 'justified')->count();

        return [
            'total' => $total,
            'present' => $present,
            'absent' => $absent,
            'late' => $late,
            'justified' => $justified,
            'rate' => $total > 0 ? round(($present + $late + $justified) / $total * 100, 1) : 0,
        ];
    }

    private function getDayStatus($records)
    {
        if ($records->isEmpty()) return 'none';
        if ($records->where('status', 'absent')->count() > 0) return 'absent';
        if ($records->where('status', 'late')->count() > 0) return 'late';
        if ($records->where('status', 'justified')->count() > 0) return 'justified';
        return 'present';
    }

    private function buildCalendarData($studentId)
    {
        $firstDay = Carbon::createFromDate($this->year, $this->month, 1);
        $lastDay = $firstDay->copy()->endOfMonth();
        $startOfWeek = $firstDay->copy()->startOfWeek();

        $records = Attendance::where('student_id', $studentId)
            ->whereMonth('date', $this->month)
            ->whereYear('date', $this->year)
            ->get()
            ->groupBy(function ($r) {
                return $r->date->format('Y-m-d');
            });

        $calendar = [];
        $current = $startOfWeek->copy();

        while ($current <= $lastDay || $current->dayOfWeek != 1) {
            $dateStr = $current->format('Y-m-d');
            $isCurrentMonth = $current->month == $this->month;
            $dayRecords = $records->get($dateStr, collect());

            $calendar[] = [
                'date' => $dateStr,
                'day' => $current->day,
                'is_current_month' => $isCurrentMonth,
                'is_today' => $current->isToday(),
                'status' => $isCurrentMonth ? $this->getDayStatus($dayRecords) : 'none',
                'count' => $dayRecords->count(),
            ];

            $current->addDay();
            if (!$isCurrentMonth && $current->month != $this->month && $current->dayOfWeek == 1) break;
        }

        return $calendar;
    }
}
