<?php

namespace App\Imports;

use App\Models\User;
use App\Models\Fee_assign;
use App\Services\FeeCalculationService;
use Maatwebsite\Excel\Concerns\ToArray;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class FeePaymentsImport implements ToArray, WithHeadingRow, WithValidation
{
    protected $results = ['imported' => 0, 'skipped' => 0, 'errors' => []];
    protected $feeCalculationService;

    protected $monthNames = [
        1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
        5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
        9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
    ];

    protected $paymentModes = [
        'cash' => 'Cash', 'dinheiro' => 'Cash',
        'bank' => 'Bank', 'banco' => 'Bank', 'transferencia' => 'Bank',
        'mobile' => 'Mobile', 'mpesa' => 'Mobile', 'emola' => 'Mobile',
        'pos' => 'POS',
    ];

    public function __construct()
    {
        $this->feeCalculationService = app(FeeCalculationService::class);
    }

    public function array(array $rows)
    {
        DB::beginTransaction();
        try {
            foreach ($rows as $row) {
                $student = User::where('student_id', $row['numero_aluno'])->first();
                if (!$student) {
                    $this->results['skipped']++;
                    continue;
                }

                $mes = intval($row['mes_01_12'] ?? $row['mes']);
                $ano = intval($row['ano']);
                $monthName = $this->monthNames[$mes] ?? null;

                if (!$monthName) {
                    $this->results['skipped']++;
                    continue;
                }

                if ($this->feeCalculationService->hasPaymentForPeriod($student, $monthName, $ano)) {
                    $this->results['skipped']++;
                    continue;
                }

                $paymentDate = Carbon::createFromFormat('d/m/Y', $row['data_pagamento_dd_mm_aaaa'] ?? $row['data_pagamento']);
                $paymentMode = $this->normalizePaymentMode($row['modo_pagamento'] ?? 'Cash');

                $this->feeCalculationService->createFeePayment($student, [
                    'month' => $monthName,
                    'year' => $ano,
                    'amount' => floatval($row['valor_mt'] ?? $row['valor']),
                    'fine' => 0,
                    'discount' => 0,
                    'payment_mode' => $paymentMode,
                    'pay_type' => strtolower($paymentMode),
                    'transaction_id' => null,
                    'reference_number' => null,
                    'payment_date' => $paymentDate,
                ]);

                $this->results['imported']++;
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            $this->results['errors'][] = $e->getMessage();
            Log::error('FeePaymentsImport failed', ['error' => $e->getMessage()]);
        }
    }

    public function rules(): array
    {
        return [
            'numero_aluno' => 'required',
            '*.mes_01_12' => 'nullable',
            '*.mes' => 'nullable',
            '*.ano' => 'required',
            '*.valor_mt' => 'nullable',
            '*.valor' => 'nullable',
        ];
    }

    public function getResults(): array
    {
        return $this->results;
    }

    private function normalizePaymentMode(string $input): string
    {
        return $this->paymentModes[strtolower(trim($input))] ?? 'Cash';
    }
}
