<?php

namespace App\Imports;

use App\Models\PaymentReference;
use App\Models\User;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\SkipsOnError;
use Maatwebsite\Excel\Concerns\SkipsErrors;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterImport;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Twilio\Rest\Client;
use Exception;

class PaymentReferencesImport implements ToModel, WithHeadingRow, WithValidation, SkipsOnError, WithEvents
{
    use SkipsErrors;

    protected $entityCode;
    protected $importedReferences = [];

    public function __construct($entityCode)
    {
        $this->entityCode = $entityCode;
    }

    /**
    * @param array $row
    *
    * Formato esperado do Excel:
    * codigo_estudante | valor | descricao | mes | ano | validade_dias | data_inicio
    */
    public function model(array $row)
    {
        // Buscar estudante pelo código
        $student = User::where('student_id', $row['codigo_estudante'])
            ->orWhere('email', $row['codigo_estudante'])
            ->first();

        if (!$student) {
            return null; // Skip se estudante não encontrado
        }

        // Gerar número de referência único
        $referenceNumber = $this->generateReferenceNumber();

        // Pegar dados do Excel
        $feeMonth = ucfirst(strtolower($row['mes_janeiro_dezembro'] ?? $row['mes'] ?? 'Janeiro'));
        $feeYear = intval($row['ano'] ?? date('Y'));
        $validityDays = intval($row['validade_dias'] ?? 30);
        $startDate = $row['data_inicio'] ?? now()->format('Y-m-d');

        // Calcular datas de validade
        $validFrom = Carbon::parse($startDate)->startOfDay();
        $expiresAt = Carbon::parse($startDate)->addDays($validityDays)->endOfDay();

        // Buscar informações da turma do estudante
        $classe = $student->classroom?->class;
        $turma = $student->classroom?->name;

        // Determina o ano acadêmico
        $academicYear = $feeYear . '-' . ($feeYear + 1);

        $reference = new PaymentReference([
            'student_id' => $student->id,
            'classe' => $classe,
            'turma' => $turma,
            'entity_code' => $this->entityCode,
            'reference_number' => $referenceNumber,
            'amount' => $row['valor_mt'] ?? $row['valor'],
            'fee_month' => $feeMonth,
            'fee_year' => $feeYear,
            'academic_year' => $academicYear,
            'status' => 'pending',
            'valid_from' => $validFrom,
            'expires_at' => $expiresAt,
        ]);

        // Armazenar referência importada para enviar SMS depois
        $this->importedReferences[] = [
            'student' => $student,
            'reference' => $reference,
        ];

        return $reference;
    }

    public function rules(): array
    {
        return [
            'codigo_estudante' => 'required',
            'valor_mt' => 'required|numeric|min:1',
        ];
    }

    private function generateReferenceNumber()
    {
        do {
            $datePart = now()->format('ymd'); // 6 dígitos
            $randomPart = str_pad(mt_rand(0, 99999), 5, '0', STR_PAD_LEFT); // 5 dígitos
            $referenceNumber = $datePart . $randomPart;

            $exists = PaymentReference::where('reference_number', $referenceNumber)->exists();
        } while ($exists);

        return $referenceNumber;
    }

    public function registerEvents(): array
    {
        return [
            AfterImport::class => function(AfterImport $event) {
                // Enviar SMS para todas as referências importadas
                $this->sendBulkSMS();
            },
        ];
    }

    private function sendBulkSMS()
    {
        if (empty($this->importedReferences)) {
            return;
        }

        $account_sid = getenv("TWILIO_SID");
        $auth_token = getenv("TWILIO_TOKEN");
        $twilio_number = getenv("TWILIO_FROM");

        if (!$account_sid || !$auth_token || !$twilio_number) {
            Log::warning('Configuração Twilio não encontrada - SMS em lote não enviado');
            return;
        }

        $client = new Client($account_sid, $auth_token);
        $sentCount = 0;
        $failedCount = 0;

        foreach ($this->importedReferences as $item) {
            $student = $item['student'];
            $reference = $item['reference'];

            if (empty($student->phone_number)) {
                continue;
            }

            try {
                $message = "COPMOZ - Ref. Pagamento\n" .
                           "Entidade: {$reference->entity_code}\n" .
                           "Ref: {$reference->reference_number}\n" .
                           "Valor: {$reference->amount} MT\n" .
                           "Validade: " . Carbon::parse($reference->expires_at)->format('d/m/Y') . "\n" .
                           "Estudante: {$student->name}";

                $client->messages->create('+258' . $student->phone_number, [
                    'from' => $twilio_number,
                    'body' => $message,
                ]);

                $sentCount++;

                Log::info('SMS de importação enviado', [
                    'student_id' => $student->id,
                    'reference' => $reference->reference_number,
                ]);

            } catch (Exception $e) {
                $failedCount++;
                Log::error('Erro ao enviar SMS de importação', [
                    'student_id' => $student->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        Log::info('SMS em lote concluído', [
            'total' => count($this->importedReferences),
            'enviados' => $sentCount,
            'falhas' => $failedCount,
        ]);
    }
}
