<?php

namespace App\Models;

use App\Traits\Uuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Note\Models\Notification;
use Spatie\Permission\Traits\HasRoles;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;

class Admin extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, Uuids, HasRoles, HasSlug, SoftDeletes;

    /**
     * stop the autoincrement
     */
    public $incrementing = false;

    /**
     * type of auto-increment
     *
     * @string
     */
    protected $keyType = 'string';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'phone_number',
        'slug',
        'email',
        'role',
        'password',
        'is_active',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the options for generating the slug.
     */
    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('name')
            ->saveSlugsTo('slug');
    }

    /**
     * notifications
     * @return MorphMany
     */
    public function notification(): MorphMany
    {
        return $this->morphMany(Notification::class, 'notification');
    }

    /**
     * Teacher subjects relationship
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function teacherSubjects()
    {
        return $this->hasMany(TeacherSubject::class, 'teacher_id');
    }

    /**
     * Admin permissions relationship
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function permissions()
    {
        return $this->hasMany(AdminPermission::class, 'admin_id');
    }

    /**
     * Check if admin has a specific permission
     * Combines role-based and custom permissions
     * @param string $permissionKey
     * @return bool
     */
    public function hasPermission($permissionKey)
    {
        // Super Admin (role 0) has all permissions
        if ($this->role == 0) {
            return true;
        }

        // Check role-based default permissions
        $rolePermissions = $this->getRoleDefaultPermissions();
        if (in_array($permissionKey, $rolePermissions)) {
            // Check if this permission was explicitly revoked
            $customPerm = AdminPermission::where('admin_id', $this->id)
                ->where('permission_key', $permissionKey)
                ->first();

            if ($customPerm && !$customPerm->granted) {
                return false; // Explicitly revoked
            }

            return true; // Has by default from role
        }

        // Check custom granted permissions
        return AdminPermission::hasPermission($this->id, $permissionKey);
    }

    /**
     * Get default permissions based on role
     * Now reads from database (role_permissions table) instead of hardcoded values
     * @return array
     */
    public function getRoleDefaultPermissions()
    {
        // Super Admin has all permissions
        if ($this->role == 0) {
            return array_keys($this->getAllPermissions());
        }

        // Get permissions from database
        $basePermissions = RolePermission::getPermissionsForRole($this->role);

        // If no permissions configured in database, return empty array
        // Admin should configure role permissions via the management interface

        // Check if user also teaches (has teacher subjects)
        // If they are both Reg Academico (1) AND teach, they also get professor permissions
        if ($this->role == 1 && $this->teacherSubjects()->exists()) {
            $professorPerms = RolePermission::getPermissionsForRole(2); // Role 2 = Professor
            $basePermissions = array_unique(array_merge($basePermissions, $professorPerms));
        }

        return $basePermissions;
    }

    /**
     * Get all available permissions
     * @return array
     */
    private function getAllPermissions()
    {
        return self::getAllPermissionsStatic();
    }

    /**
     * Get all available permissions (static version)
     * @return array
     */
    public static function getAllPermissionsStatic()
    {
        $all = [];
        foreach (AdminPermission::availablePermissions() as $group) {
            $all = array_merge($all, $group['permissions']);
        }
        return $all;
    }

    /**
     * Get all granted permissions for this admin
     * @return array
     */
    public function getPermissions()
    {
        // Super Admin has all
        if ($this->role == 0) {
            return array_keys($this->getAllPermissions());
        }

        // Combine role defaults + custom grants - custom revokes
        $rolePerms = $this->getRoleDefaultPermissions();
        $customPerms = AdminPermission::where('admin_id', $this->id)->get();

        $finalPerms = $rolePerms;

        foreach ($customPerms as $perm) {
            if ($perm->granted && !in_array($perm->permission_key, $finalPerms)) {
                $finalPerms[] = $perm->permission_key;
            } elseif (!$perm->granted && in_array($perm->permission_key, $finalPerms)) {
                $finalPerms = array_diff($finalPerms, [$perm->permission_key]);
            }
        }

        return array_values($finalPerms);
    }

    /**
     * Get role name
     * @return string
     */
    public function getRoleName()
    {
        return match($this->role) {
            0 => 'Super Admin',
            1 => 'Reg Academico',
            2 => 'Professor',
            3 => 'Diretor',
            4 => 'Admin',
            5 => 'Contabilidade',
            6 => 'Secretaria',
            default => 'Desconhecido',
        };
    }

    /**
     * Check if admin is PCA (Super Admin)
     * @return bool
     */
    public function isPca()
    {
        return $this->role == 0;
    }

    public function recordedAttendances()
    {
        return $this->hasMany(Attendance::class, 'recorded_by');
    }
}
