<?php

namespace App\Models;

use App\Traits\Uuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class CertificateBatch extends Model
{
    use HasFactory, Uuids, SoftDeletes;

    public $incrementing = false;
    protected $keyType = 'string';

    protected $fillable = [
        'batch_number',
        'description',
        'classroom_id',
        'academic_year',
        'status',
        'total_students',
        'processed_count',
        'success_count',
        'failed_count',
        'failed_students',
        'zip_path',
        'zip_generated_at',
        'created_by',
    ];

    protected $casts = [
        'failed_students' => 'array',
        'zip_generated_at' => 'datetime',
        'academic_year' => 'integer',
        'total_students' => 'integer',
        'processed_count' => 'integer',
        'success_count' => 'integer',
        'failed_count' => 'integer',
    ];

    /**
     * Relationships
     */
    public function classroom(): BelongsTo
    {
        return $this->belongsTo(Classroom::class, 'classroom_id');
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    /**
     * Scopes
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeProcessing($query)
    {
        return $query->where('status', 'processing');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    public function scopeForYear($query, $year)
    {
        return $query->where('academic_year', $year);
    }

    /**
     * Helpers
     */
    public function getProgressPercentage(): int
    {
        if ($this->total_students === 0) {
            return 0;
        }
        return (int) round(($this->processed_count / $this->total_students) * 100);
    }

    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    public function isProcessing(): bool
    {
        return $this->status === 'processing';
    }

    public function hasZip(): bool
    {
        return !empty($this->zip_path);
    }

    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'pending' => '<span class="badge bg-warning">Pendente</span>',
            'processing' => '<span class="badge bg-info">Processando</span>',
            'completed' => '<span class="badge bg-success">Concluido</span>',
            'failed' => '<span class="badge bg-danger">Falhou</span>',
            default => '<span class="badge bg-secondary">Desconhecido</span>',
        };
    }

    public function getProgressBarAttribute(): string
    {
        $percentage = $this->getProgressPercentage();
        $color = match(true) {
            $percentage >= 100 => 'success',
            $percentage >= 50 => 'info',
            $percentage >= 25 => 'warning',
            default => 'danger',
        };

        return '<div class="progress" style="height: 20px;">
            <div class="progress-bar bg-' . $color . '" role="progressbar"
                style="width: ' . $percentage . '%;"
                aria-valuenow="' . $percentage . '" aria-valuemin="0" aria-valuemax="100">
                ' . $percentage . '%
            </div>
        </div>';
    }
}
