<?php

// app/Models/FeeAssign.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class FeeAssign extends Model
{
    use HasFactory;

    protected $table = 'fee_assign';

    protected $fillable = [
        'student_id',
        'fee_group_id',
        'status',
        'day',
        'month',
        'year',
        'pay_type',
        'pay_id',
        'pay_day',
        'pay_month',
        'pay_year',
        'discount',
        'fine',
        'amount'
    ];

    protected $casts = [
        'discount' => 'decimal:2',
        'fine' => 'decimal:2',
        'amount' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Relacionamento com o usuário (estudante)
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'student_id');
    }

    /**
     * Relacionamento com a estrutura de taxas
     */
    public function feeStructure(): BelongsTo
    {
        return $this->belongsTo(FeeStructure::class, 'fee_group_id');
    }

    /**
     * Scope para pagamentos pagos
     */
    public function scopePaid($query)
    {
        return $query->where('status', 'paid');
    }

    /**
     * Scope para pagamentos pendentes
     */
    public function scopeUnpaid($query)
    {
        return $query->where('status', 'unpaid');
    }

    /**
     * Scope para pagamentos atrasados
     */
    public function scopeOverdue($query)
    {
        return $query->where('status', 'overdue')
                     ->orWhere(function($q) {
                         $q->where('status', 'unpaid')
                           ->whereNotNull('pay_day')
                           ->where('pay_day', '<', now()->format('Y-m-d'));
                     });
    }

    /**
     * Calcula o valor total (amount - discount + fine)
     */
    public function getTotalAmountAttribute()
    {
        return ($this->amount ?? 0) - ($this->discount ?? 0) + ($this->fine ?? 0);
    }

    /**
     * Verifica se o pagamento está atrasado
     */
    public function getIsOverdueAttribute()
    {
        if ($this->status === 'paid') {
            return false;
        }

        if ($this->pay_day && $this->pay_day < now()->format('Y-m-d')) {
            return true;
        }

        return false;
    }

    /**
     * Formata o status para exibição
     */
    public function getFormattedStatusAttribute()
    {
        $statuses = [
            'paid' => 'Pago',
            'unpaid' => 'Pendente',
            'partial' => 'Parcial',
            'overdue' => 'Atrasado'
        ];

        return $statuses[$this->status] ?? $this->status;
    }

    /**
     * Formata o método de pagamento
     */
    public function getFormattedPayTypeAttribute()
    {
        $types = [
            'Cash' => 'Dinheiro',
            'Bank' => 'Transferência Bancária',
            'Online' => 'Online',
            'Mobile' => 'Mobile Money'
        ];

        return $types[$this->pay_type] ?? $this->pay_type;
    }
}
