<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class PermissionDefinition extends Model
{
    protected $fillable = [
        'key',
        'label',
        'description',
        'group_id',
        'sort_order',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Get the group this permission belongs to
     */
    public function group()
    {
        return $this->belongsTo(PermissionGroup::class, 'group_id');
    }

    /**
     * Get all permissions as key => label array (cached)
     */
    public static function getAllAsArray(): array
    {
        return Cache::remember('available_permissions', 3600, function () {
            return self::where('is_active', true)
                ->pluck('label', 'key')
                ->toArray();
        });
    }

    /**
     * Get all permission keys
     */
    public static function getAllKeys(): array
    {
        return array_keys(self::getAllAsArray());
    }

    /**
     * Get permissions formatted for UI (grouped)
     */
    public static function getGroupedForUI(): array
    {
        $groups = PermissionGroup::where('is_active', true)
            ->orderBy('sort_order')
            ->orderBy('label')
            ->with(['activePermissions'])
            ->get();

        $result = [];
        foreach ($groups as $group) {
            $permissions = [];
            foreach ($group->activePermissions as $perm) {
                $permissions[$perm->key] = $perm->label;
            }

            if (!empty($permissions)) {
                $result[$group->key] = [
                    'label' => $group->label,
                    'permissions' => $permissions,
                ];
            }
        }

        return $result;
    }

    /**
     * Clear permission cache
     */
    public static function clearCache(): void
    {
        Cache::forget('available_permissions');
        PermissionGroup::clearCache();
    }

    /**
     * Seed default permissions
     */
    public static function seedDefaults(): void
    {
        // First seed groups
        PermissionGroup::seedDefaults();

        // Get group IDs
        $groups = PermissionGroup::pluck('id', 'key')->toArray();

        $permissions = [
            // POS Permissions
            ['key' => 'pos.access', 'label' => 'Acesso ao POS', 'group_id' => $groups['pos'], 'sort_order' => 1],
            ['key' => 'pos.dashboard', 'label' => 'Ver Dashboard POS', 'group_id' => $groups['pos'], 'sort_order' => 2],
            ['key' => 'pos.sales', 'label' => 'Realizar Vendas', 'group_id' => $groups['pos'], 'sort_order' => 3],
            ['key' => 'pos.sales_report', 'label' => 'Ver Relatório de Vendas', 'group_id' => $groups['pos'], 'sort_order' => 4],
            ['key' => 'pos.products.view', 'label' => 'Ver Produtos', 'group_id' => $groups['pos'], 'sort_order' => 5],
            ['key' => 'pos.products.manage', 'label' => 'Gerir Produtos (Adicionar/Editar/Excluir)', 'group_id' => $groups['pos'], 'sort_order' => 6],
            ['key' => 'pos.categories.view', 'label' => 'Ver Categorias', 'group_id' => $groups['pos'], 'sort_order' => 7],
            ['key' => 'pos.categories.manage', 'label' => 'Gerir Categorias', 'group_id' => $groups['pos'], 'sort_order' => 8],
            ['key' => 'pos.settings', 'label' => 'Configurações POS', 'group_id' => $groups['pos'], 'sort_order' => 9],

            // Students Permissions
            ['key' => 'students.view', 'label' => 'Ver Estudantes', 'group_id' => $groups['students'], 'sort_order' => 1],
            ['key' => 'students.add', 'label' => 'Adicionar Estudantes', 'group_id' => $groups['students'], 'sort_order' => 2],
            ['key' => 'students.edit', 'label' => 'Editar Estudantes', 'group_id' => $groups['students'], 'sort_order' => 3],
            ['key' => 'students.delete', 'label' => 'Excluir Estudantes', 'group_id' => $groups['students'], 'sort_order' => 4],
            ['key' => 'students.export', 'label' => 'Exportar Dados', 'group_id' => $groups['students'], 'sort_order' => 5],

            // Teachers Permissions
            ['key' => 'teachers.view', 'label' => 'Ver Professores', 'group_id' => $groups['teachers'], 'sort_order' => 1],
            ['key' => 'teachers.add', 'label' => 'Adicionar Professores', 'group_id' => $groups['teachers'], 'sort_order' => 2],
            ['key' => 'teachers.edit', 'label' => 'Editar Professores', 'group_id' => $groups['teachers'], 'sort_order' => 3],
            ['key' => 'teachers.delete', 'label' => 'Excluir Professores', 'group_id' => $groups['teachers'], 'sort_order' => 4],
            ['key' => 'teachers.assign', 'label' => 'Atribuir Disciplinas', 'group_id' => $groups['teachers'], 'sort_order' => 5],

            // Classes Permissions
            ['key' => 'classes.view', 'label' => 'Ver Turmas', 'group_id' => $groups['classes'], 'sort_order' => 1],
            ['key' => 'classes.add', 'label' => 'Adicionar Turmas', 'group_id' => $groups['classes'], 'sort_order' => 2],
            ['key' => 'classes.edit', 'label' => 'Editar Turmas', 'group_id' => $groups['classes'], 'sort_order' => 3],
            ['key' => 'classes.delete', 'label' => 'Excluir Turmas', 'group_id' => $groups['classes'], 'sort_order' => 4],

            // Exams Permissions
            ['key' => 'exams.view', 'label' => 'Ver Notas', 'group_id' => $groups['exams'], 'sort_order' => 1],
            ['key' => 'exams.add', 'label' => 'Adicionar Notas', 'group_id' => $groups['exams'], 'sort_order' => 2],
            ['key' => 'exams.edit', 'label' => 'Editar Notas', 'group_id' => $groups['exams'], 'sort_order' => 3],
            ['key' => 'exams.delete', 'label' => 'Excluir Notas', 'group_id' => $groups['exams'], 'sort_order' => 4],
            ['key' => 'exams.reports', 'label' => 'Gerar Relatórios', 'group_id' => $groups['exams'], 'sort_order' => 5],

            // Payments Permissions
            ['key' => 'payments.view', 'label' => 'Ver Pagamentos', 'group_id' => $groups['payments'], 'sort_order' => 1],
            ['key' => 'payments.collect', 'label' => 'Cobrar Pagamentos', 'group_id' => $groups['payments'], 'sort_order' => 2],
            ['key' => 'payments.manage', 'label' => 'Gerir Tipos de Taxas', 'group_id' => $groups['payments'], 'sort_order' => 3],
            ['key' => 'payments.reports', 'label' => 'Relatórios de Pagamento', 'group_id' => $groups['payments'], 'sort_order' => 4],
            ['key' => 'payments.import', 'label' => 'Importar Pagamentos (Excel)', 'group_id' => $groups['payments'], 'sort_order' => 5],

            // Library Permissions
            ['key' => 'library.view', 'label' => 'Ver Biblioteca', 'group_id' => $groups['library'], 'sort_order' => 1],
            ['key' => 'library.add', 'label' => 'Adicionar Livros', 'group_id' => $groups['library'], 'sort_order' => 2],
            ['key' => 'library.edit', 'label' => 'Editar Livros', 'group_id' => $groups['library'], 'sort_order' => 3],
            ['key' => 'library.delete', 'label' => 'Excluir Livros', 'group_id' => $groups['library'], 'sort_order' => 4],

            // System Permissions
            ['key' => 'system.staff.manage', 'label' => 'Gerir Staff', 'group_id' => $groups['system'], 'sort_order' => 1],
            ['key' => 'system.permissions', 'label' => 'Gerir Permissões', 'group_id' => $groups['system'], 'sort_order' => 2],
            ['key' => 'system.settings', 'label' => 'Configurações do Sistema', 'group_id' => $groups['system'], 'sort_order' => 3],

            // SMS Permissions
            ['key' => 'sms.send', 'label' => 'Enviar SMS', 'group_id' => $groups['sms'], 'sort_order' => 1],
            ['key' => 'sms.view', 'label' => 'Ver SMS Enviados', 'group_id' => $groups['sms'], 'sort_order' => 2],
            ['key' => 'sms.settings', 'label' => 'Configurações SMS', 'group_id' => $groups['sms'], 'sort_order' => 3],

            // Certificates Permissions
            ['key' => 'certificates.view', 'label' => 'Ver Certificados', 'group_id' => $groups['certificates'], 'sort_order' => 1],
            ['key' => 'certificates.generate', 'label' => 'Gerar Certificados', 'group_id' => $groups['certificates'], 'sort_order' => 2],
            ['key' => 'certificates.download', 'label' => 'Baixar Certificados', 'group_id' => $groups['certificates'], 'sort_order' => 3],
            ['key' => 'certificates.batch', 'label' => 'Gerar em Lote', 'group_id' => $groups['certificates'], 'sort_order' => 4],
            ['key' => 'certificates.revoke', 'label' => 'Revogar Certificados', 'group_id' => $groups['certificates'], 'sort_order' => 5],

            // Student Cards Permissions
            ['key' => 'student_cards.view', 'label' => 'Ver Cartões', 'group_id' => $groups['student_cards'], 'sort_order' => 1],
            ['key' => 'student_cards.generate', 'label' => 'Gerar Cartões', 'group_id' => $groups['student_cards'], 'sort_order' => 2],
            ['key' => 'student_cards.download', 'label' => 'Baixar Cartões', 'group_id' => $groups['student_cards'], 'sort_order' => 3],
            ['key' => 'student_cards.batch', 'label' => 'Gerar em Lote', 'group_id' => $groups['student_cards'], 'sort_order' => 4],

            // Attendance Permissions
            ['key' => 'attendance.mark', 'label' => 'Marcar Presença', 'group_id' => $groups['attendance'], 'sort_order' => 1],
            ['key' => 'attendance.view', 'label' => 'Ver Registos de Presença', 'group_id' => $groups['attendance'], 'sort_order' => 2],
            ['key' => 'attendance.edit', 'label' => 'Editar Presença', 'group_id' => $groups['attendance'], 'sort_order' => 3],
            ['key' => 'attendance.delete', 'label' => 'Excluir Presença', 'group_id' => $groups['attendance'], 'sort_order' => 4],
            ['key' => 'attendance.report', 'label' => 'Relatórios de Presença', 'group_id' => $groups['attendance'], 'sort_order' => 5],

            // Secretary Permissions
            ['key' => 'secretary.pre_enrollments.view', 'label' => 'Ver Pré-inscrições', 'group_id' => $groups['secretary'], 'sort_order' => 1],
            ['key' => 'secretary.pre_enrollments.manage', 'label' => 'Gerir Pré-inscrições (Confirmar/Anular)', 'group_id' => $groups['secretary'], 'sort_order' => 2],
            ['key' => 'secretary.enrollment.create', 'label' => 'Realizar Matrículas', 'group_id' => $groups['secretary'], 'sort_order' => 3],

            // Renewals Permissions
            ['key' => 'renewals.view', 'label' => 'Ver Renovações de Matrícula', 'group_id' => $groups['renewals'], 'sort_order' => 1],
            ['key' => 'renewals.manage', 'label' => 'Confirmar / Cancelar Renovações', 'group_id' => $groups['renewals'], 'sort_order' => 2],
            ['key' => 'renewals.settings', 'label' => 'Configurar Período de Renovação', 'group_id' => $groups['renewals'], 'sort_order' => 3],

            // Communications Permissions (Comunicados e Notificações)
            ['key' => 'communications.send_notification', 'label' => 'Enviar Notificação', 'group_id' => $groups['communications'], 'sort_order' => 1],
            ['key' => 'communications.add_notice', 'label' => 'Adicionar Comunicado', 'group_id' => $groups['communications'], 'sort_order' => 2],
            ['key' => 'communications.view_notices', 'label' => 'Ver Comunicados', 'group_id' => $groups['communications'], 'sort_order' => 3],
            ['key' => 'communications.payment_reminders', 'label' => 'Lembretes de Pagamento', 'group_id' => $groups['communications'], 'sort_order' => 4],

            // Documents Permissions (Documentos e Matrizes)
            ['key' => 'documents.add_template', 'label' => 'Adicionar Matriz', 'group_id' => $groups['documents'], 'sort_order' => 1],
            ['key' => 'documents.view_templates', 'label' => 'Ver Matrizes', 'group_id' => $groups['documents'], 'sort_order' => 2],
            ['key' => 'documents.certificates', 'label' => 'Gerir Certificados', 'group_id' => $groups['documents'], 'sort_order' => 3],
            ['key' => 'documents.student_cards', 'label' => 'Gerir Cartões de Estudante', 'group_id' => $groups['documents'], 'sort_order' => 4],
        ];

        foreach ($permissions as $perm) {
            self::updateOrCreate(['key' => $perm['key']], $perm);
        }

        self::clearCache();
    }

    /**
     * Check if permissions have been seeded
     */
    public static function isSeeded(): bool
    {
        return self::count() > 0;
    }
}
