<?php

namespace App\Models;

use App\Traits\Uuids;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class SystemSetting extends Model
{
    use Uuids;

    protected $table = 'system_settings';

    protected $fillable = [
        'group',
        'key',
        'value',
        'type',
        'description',
    ];

    /**
     * Get a setting value with type casting
     */
    public static function get(string $key, $default = null, ?string $group = null)
    {
        $cacheKey = $group ? "system_setting_{$group}_{$key}" : "system_setting_{$key}";

        return Cache::remember($cacheKey, 3600, function () use ($key, $default, $group) {
            $query = self::where('key', $key);

            if ($group) {
                $query->where('group', $group);
            }

            $setting = $query->first();

            if (!$setting) {
                return $default;
            }

            return self::castValue($setting->value, $setting->type);
        });
    }

    /**
     * Set a setting value
     */
    public static function set(string $key, $value, string $type = 'string', ?string $group = null, ?string $description = null)
    {
        $data = [
            'value' => is_array($value) ? json_encode($value) : $value,
            'type' => $type,
        ];

        if ($group) {
            $data['group'] = $group;
        }

        if ($description) {
            $data['description'] = $description;
        }

        $conditions = ['key' => $key];
        if ($group) {
            $conditions['group'] = $group;
        }

        $setting = self::updateOrCreate($conditions, $data);

        $cacheKey = $group ? "system_setting_{$group}_{$key}" : "system_setting_{$key}";
        Cache::forget($cacheKey);

        return $setting;
    }

    /**
     * Get all settings for a group
     */
    public static function getGroup(string $group): array
    {
        return Cache::remember("system_settings_group_{$group}", 3600, function () use ($group) {
            $settings = self::where('group', $group)->get();

            $result = [];
            foreach ($settings as $setting) {
                $result[$setting->key] = self::castValue($setting->value, $setting->type);
            }

            return $result;
        });
    }

    /**
     * Clear cache for a group
     */
    public static function clearGroupCache(string $group): void
    {
        Cache::forget("system_settings_group_{$group}");

        $settings = self::where('group', $group)->get();
        foreach ($settings as $setting) {
            Cache::forget("system_setting_{$group}_{$setting->key}");
        }
    }

    /**
     * Cast value based on type
     */
    private static function castValue($value, $type)
    {
        return match ($type) {
            'boolean' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'number', 'integer', 'float' => is_numeric($value) ? (float) $value : 0,
            'json', 'array' => is_string($value) ? json_decode($value, true) : $value,
            default => $value,
        };
    }
}
