<?php

namespace App\Services;

use App\Models\PaymentReference;
use App\Models\User;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class PaymentReceiptService
{
    /**
     * Gera o comprovativo de pagamento em PDF
     *
     * @param PaymentReference $payment
     * @return string|null Retorna o caminho do PDF ou null em caso de erro
     */
    public function generateReceipt(PaymentReference $payment): ?string
    {
        try {
            // Verificar se o pagamento está pago
            if ($payment->status !== 'paid') {
                Log::warning('Tentativa de gerar comprovativo para pagamento não pago', [
                    'payment_id' => $payment->id,
                    'status' => $payment->status
                ]);
                return null;
            }

            $student = $payment->student;
            if (!$student) {
                Log::error('Estudante não encontrado para o pagamento', [
                    'payment_id' => $payment->id
                ]);
                return null;
            }

            // Preparar dados para o PDF
            $data = $this->prepareReceiptData($payment, $student);

            // Gerar PDF
            $pdf = Pdf::loadView('pdf.payment-receipt', $data)
                ->setOptions([
                    'isRemoteEnabled' => false,
                    'isHtml5ParserEnabled' => true,
                    'isFontSubsettingEnabled' => true,
                ])
                ->setPaper('a4', 'portrait');

            // Definir nome do arquivo
            $fileName = $this->generateFileName($student, $payment);
            $filePath = "receipts/{$fileName}";

            // Salvar PDF
            $pdfContent = $pdf->output();
            Storage::disk('public')->put($filePath, $pdfContent);

            Log::info('Comprovativo de pagamento gerado com sucesso', [
                'payment_id' => $payment->id,
                'student_id' => $student->id,
                'file_path' => $filePath
            ]);

            return $filePath;

        } catch (\Exception $e) {
            Log::error('Erro ao gerar comprovativo de pagamento', [
                'payment_id' => $payment->id ?? null,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return null;
        }
    }

    /**
     * Prepara os dados para o PDF do comprovativo
     */
    private function prepareReceiptData(PaymentReference $payment, User $student): array
    {
        // Formatar a referência
        $reference = $payment->reference_number;
        $reference_formatted = $this->formatReference($reference);

        // Calcular valores
        $amount_paid = $payment->paid_amount ?? $payment->amount;
        $base_amount = $payment->amount ?? 0;
        $fine_amount = 0;
        $discount_amount = 0;

        // Se existe bmepsTransaction, usar os dados de lá
        if ($payment->bmepsTransactions && $payment->bmepsTransactions->isNotEmpty()) {
            $transaction = $payment->bmepsTransactions->first();
            $transaction_id = $transaction->transaction_id ?? null;
            $payment_method = $this->getPaymentMethodName($transaction->payment_method ?? null);
        } else {
            $transaction_id = null;
            $payment_method = 'Pagamento via Referência';
        }

        return [
            'student' => $student,
            'payment' => $payment,
            'reference_formatted' => $reference_formatted,
            'amount_paid' => $amount_paid,
            'base_amount' => $base_amount,
            'fine_amount' => $fine_amount,
            'discount_amount' => $discount_amount,
            'paid_at' => $payment->paid_at ? Carbon::parse($payment->paid_at) : now(),
            'transaction_id' => $transaction_id,
            'payment_method' => $payment_method,
            'notes' => $payment->note ?? null,
        ];
    }

    /**
     * Formata a referência para exibição (XXX XXX XXX XX)
     */
    private function formatReference(string $reference): string
    {
        $ref = preg_replace('/\D/', '', $reference);

        if (strlen($ref) === 11) {
            return substr($ref, 0, 3) . ' ' . substr($ref, 3, 3) . ' ' . substr($ref, 6, 3) . ' ' . substr($ref, 9, 2);
        }

        return $reference;
    }

    /**
     * Obtém o nome do método de pagamento
     */
    private function getPaymentMethodName(?string $method): string
    {
        $methods = [
            'mpesa' => 'M-Pesa',
            'emola' => 'e-Mola',
            'atm' => 'ATM',
            'bank_transfer' => 'Transferência Bancária',
            'cash' => 'Dinheiro',
        ];

        return $methods[strtolower($method ?? '')] ?? 'Pagamento via Referência';
    }

    /**
     * Gera o nome do arquivo do comprovativo
     */
    private function generateFileName(User $student, PaymentReference $payment): string
    {
        $studentName = \Illuminate\Support\Str::slug($student->name);
        $reference = $payment->reference_number;
        $date = now()->format('d-m-Y_H-i');

        return "{$studentName}_comprovativo_{$reference}_{$date}.pdf";
    }

    /**
     * Verifica se o comprovativo já foi gerado
     */
    public function receiptExists(PaymentReference $payment): bool
    {
        $student = $payment->student;
        if (!$student) return false;

        $pattern = \Illuminate\Support\Str::slug($student->name) . '_comprovativo_' . $payment->reference_number;
        $files = Storage::disk('public')->files('receipts');

        foreach ($files as $file) {
            if (str_contains($file, $pattern)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Obtém o caminho do comprovativo se existir
     */
    public function getReceiptPath(PaymentReference $payment): ?string
    {
        $student = $payment->student;
        if (!$student) return null;

        $pattern = \Illuminate\Support\Str::slug($student->name) . '_comprovativo_' . $payment->reference_number;
        $files = Storage::disk('public')->files('receipts');

        foreach ($files as $file) {
            if (str_contains($file, $pattern)) {
                return $file;
            }
        }

        return null;
    }

    /**
     * Baixa ou gera o comprovativo
     */
    public function downloadReceipt(PaymentReference $payment): array
    {
        try {
            // Verificar se já existe
            $existingPath = $this->getReceiptPath($payment);

            if ($existingPath && Storage::disk('public')->exists($existingPath)) {
                return [
                    'success' => true,
                    'path' => $existingPath,
                    'url' => Storage::disk('public')->url($existingPath)
                ];
            }

            // Gerar novo comprovativo
            $path = $this->generateReceipt($payment);

            if ($path) {
                return [
                    'success' => true,
                    'path' => $path,
                    'url' => Storage::disk('public')->url($path)
                ];
            }

            return [
                'success' => false,
                'message' => 'Erro ao gerar comprovativo'
            ];

        } catch (\Exception $e) {
            Log::error('Erro ao baixar comprovativo', [
                'payment_id' => $payment->id,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Erro ao processar comprovativo: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Gera o recibo oficial em PDF
     *
     * @param PaymentReference $payment
     * @return string|null Retorna o caminho do PDF ou null em caso de erro
     */
    public function generateOfficialReceipt(PaymentReference $payment): ?string
    {
        try {
            // Verificar se o pagamento está pago
            if ($payment->status !== 'paid') {
                Log::warning('Tentativa de gerar recibo para pagamento não pago', [
                    'payment_id' => $payment->id,
                    'status' => $payment->status
                ]);
                return null;
            }

            $student = $payment->student;
            if (!$student) {
                Log::error('Estudante não encontrado para o pagamento', [
                    'payment_id' => $payment->id
                ]);
                return null;
            }

            // Preparar dados para o PDF
            $data = $this->prepareReceiptData($payment, $student);

            // Gerar PDF
            $pdf = Pdf::loadView('pdf.receipt-official', $data)
                ->setOptions([
                    'isRemoteEnabled' => false,
                    'isHtml5ParserEnabled' => true,
                    'isFontSubsettingEnabled' => true,
                ])
                ->setPaper('a4', 'portrait');

            // Definir nome do arquivo
            $fileName = $this->generateOfficialReceiptFileName($student, $payment);
            $filePath = "receipts/{$fileName}";

            // Salvar PDF
            $pdfContent = $pdf->output();
            Storage::disk('public')->put($filePath, $pdfContent);

            Log::info('Recibo oficial gerado com sucesso', [
                'payment_id' => $payment->id,
                'student_id' => $student->id,
                'file_path' => $filePath
            ]);

            return $filePath;

        } catch (\Exception $e) {
            Log::error('Erro ao gerar recibo oficial', [
                'payment_id' => $payment->id ?? null,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return null;
        }
    }

    /**
     * Gera o nome do arquivo do recibo oficial
     */
    private function generateOfficialReceiptFileName(User $student, PaymentReference $payment): string
    {
        $studentName = \Illuminate\Support\Str::slug($student->name);
        $reference = $payment->reference_number;
        $date = now()->format('d-m-Y_H-i');

        return "{$studentName}_recibo_oficial_{$reference}_{$date}.pdf";
    }

    /**
     * Obtém o caminho do recibo oficial se existir
     */
    public function getOfficialReceiptPath(PaymentReference $payment): ?string
    {
        $student = $payment->student;
        if (!$student) return null;

        $pattern = \Illuminate\Support\Str::slug($student->name) . '_recibo_oficial_' . $payment->reference_number;
        $files = Storage::disk('public')->files('receipts');

        foreach ($files as $file) {
            if (str_contains($file, $pattern)) {
                return $file;
            }
        }

        return null;
    }

    /**
     * Baixa ou gera o recibo oficial
     */
    public function downloadOfficialReceipt(PaymentReference $payment): array
    {
        try {
            // Verificar se já existe
            $existingPath = $this->getOfficialReceiptPath($payment);

            if ($existingPath && Storage::disk('public')->exists($existingPath)) {
                return [
                    'success' => true,
                    'path' => $existingPath,
                    'url' => Storage::disk('public')->url($existingPath)
                ];
            }

            // Gerar novo recibo
            $path = $this->generateOfficialReceipt($payment);

            if ($path) {
                return [
                    'success' => true,
                    'path' => $path,
                    'url' => Storage::disk('public')->url($path)
                ];
            }

            return [
                'success' => false,
                'message' => 'Erro ao gerar recibo oficial'
            ];

        } catch (\Exception $e) {
            Log::error('Erro ao baixar recibo oficial', [
                'payment_id' => $payment->id,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Erro ao processar recibo oficial: ' . $e->getMessage()
            ];
        }
    }
}
