<?php

namespace App\Services;

use App\Models\ReceiptNumber;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReceiptNumberService
{
    /**
     * Gera ou recupera número de recibo único e sequencial
     *
     * @param string $receiptType 'fee_assign' ou 'payment_reference'
     * @param int $recordId ID do registro original
     * @param string $issuedBy 'admin' ou 'student'
     * @param int|null $userId ID do usuário que gerou (opcional)
     * @return string Número do recibo (ex: 000001/2025)
     */
    public static function getOrGenerateReceiptNumber(
        string $receiptType,
        int $recordId,
        string $issuedBy = 'system',
        ?int $userId = null
    ): string {
        // Verificar se já existe recibo para este registro
        $existingReceipt = ReceiptNumber::where('receipt_type', $receiptType)
            ->where('record_id', $recordId)
            ->first();

        if ($existingReceipt) {
            return $existingReceipt->receipt_number;
        }

        // Gerar novo número de recibo
        return self::generateReceiptNumber($receiptType, $recordId, $issuedBy, $userId);
    }

    /**
     * Gera um novo número de recibo sequencial
     *
     * @param string $receiptType
     * @param int $recordId
     * @param string $issuedBy
     * @param int|null $userId
     * @return string
     * @throws \Exception
     */
    public static function generateReceiptNumber(
        string $receiptType,
        int $recordId,
        string $issuedBy = 'system',
        ?int $userId = null
    ): string {
        return DB::transaction(function () use ($receiptType, $recordId, $issuedBy, $userId) {
            $currentYear = now()->year;

            // Buscar último número do ano atual com lock para evitar duplicações
            $lastReceipt = ReceiptNumber::whereYear('issued_at', $currentYear)
                ->lockForUpdate()
                ->orderBy('id', 'desc')
                ->first();

            // Calcular próximo número
            if ($lastReceipt) {
                // Extrair número da string "000123/2025"
                $parts = explode('/', $lastReceipt->receipt_number);
                $lastNumber = (int) $parts[0];
                $nextNumber = $lastNumber + 1;
            } else {
                // Primeiro recibo do ano
                $nextNumber = 1;
            }

            // Formatar número do recibo (6 dígitos + ano)
            $receiptNumber = str_pad($nextNumber, 6, '0', STR_PAD_LEFT) . '/' . $currentYear;

            // Criar registro
            $receipt = ReceiptNumber::create([
                'receipt_type' => $receiptType,
                'record_id' => $recordId,
                'receipt_number' => $receiptNumber,
                'issued_at' => now(),
                'issued_by' => $issuedBy,
                'user_id' => $userId,
            ]);

            Log::info('Novo número de recibo gerado', [
                'receipt_number' => $receiptNumber,
                'receipt_type' => $receiptType,
                'record_id' => $recordId,
                'issued_by' => $issuedBy,
                'user_id' => $userId,
            ]);

            return $receiptNumber;
        });
    }

    /**
     * Obter próximo número de recibo (preview, sem criar)
     *
     * @return string
     */
    public static function getNextReceiptNumber(): string
    {
        $currentYear = now()->year;

        $lastReceipt = ReceiptNumber::whereYear('issued_at', $currentYear)
            ->orderBy('id', 'desc')
            ->first();

        if ($lastReceipt) {
            $parts = explode('/', $lastReceipt->receipt_number);
            $lastNumber = (int) $parts[0];
            $nextNumber = $lastNumber + 1;
        } else {
            $nextNumber = 1;
        }

        return str_pad($nextNumber, 6, '0', STR_PAD_LEFT) . '/' . $currentYear;
    }

    /**
     * Obter estatísticas de recibos
     *
     * @param int|null $year
     * @return array
     */
    public static function getStatistics(?int $year = null): array
    {
        $year = $year ?? now()->year;

        $query = ReceiptNumber::whereYear('issued_at', $year);

        return [
            'year' => $year,
            'total_receipts' => $query->count(),
            'by_type' => [
                'fee_assign' => (clone $query)->where('receipt_type', 'fee_assign')->count(),
                'payment_reference' => (clone $query)->where('receipt_type', 'payment_reference')->count(),
            ],
            'by_issuer' => [
                'admin' => (clone $query)->where('issued_by', 'admin')->count(),
                'student' => (clone $query)->where('issued_by', 'student')->count(),
                'system' => (clone $query)->where('issued_by', 'system')->count(),
            ],
            'last_receipt_number' => (clone $query)->orderBy('id', 'desc')->value('receipt_number'),
            'next_receipt_number' => self::getNextReceiptNumber(),
        ];
    }

    /**
     * Verificar se número de recibo já existe
     *
     * @param string $receiptNumber
     * @return bool
     */
    public static function exists(string $receiptNumber): bool
    {
        return ReceiptNumber::where('receipt_number', $receiptNumber)->exists();
    }

    /**
     * Buscar recibo por número
     *
     * @param string $receiptNumber
     * @return ReceiptNumber|null
     */
    public static function findByNumber(string $receiptNumber): ?ReceiptNumber
    {
        return ReceiptNumber::where('receipt_number', $receiptNumber)->first();
    }
}
