<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('certificates', function (Blueprint $table) {
            $table->uuid('id')->primary();

            // Student reference
            $table->uuid('student_id');
            $table->foreign('student_id')->references('id')->on('users')->onDelete('cascade');

            // Classroom reference (at time of graduation)
            $table->uuid('classroom_id');
            $table->foreign('classroom_id')->references('id')->on('classrooms');

            // Certificate identification
            $table->string('certificate_number')->unique();
            $table->string('verification_code', 64)->unique();
            $table->string('verification_hash', 128)->unique();

            // Academic information (snapshot at time of certificate generation)
            $table->integer('academic_year');
            $table->string('class_level')->default('12');
            $table->string('turma_name');
            $table->decimal('final_average', 5, 2);
            $table->json('subject_grades');

            // Student info snapshot (in case student record changes later)
            $table->string('student_name');
            $table->string('student_id_number');
            $table->date('student_dob')->nullable();
            $table->string('student_sex')->nullable();

            // Status
            $table->enum('status', ['draft', 'issued', 'revoked'])->default('draft');
            $table->timestamp('issued_at')->nullable();
            $table->timestamp('revoked_at')->nullable();
            $table->text('revocation_reason')->nullable();

            // Audit trail
            $table->uuid('issued_by')->nullable();
            $table->foreign('issued_by')->references('id')->on('admins')->nullOnDelete();
            $table->uuid('revoked_by')->nullable();
            $table->foreign('revoked_by')->references('id')->on('admins')->nullOnDelete();

            // PDF storage
            $table->string('pdf_path')->nullable();

            $table->softDeletes();
            $table->timestamps();

            // Indexes for faster queries
            $table->index(['academic_year', 'class_level']);
            $table->index(['student_id', 'academic_year']);
            $table->index('verification_code');
            $table->index('certificate_number');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('certificates');
    }
};
