@props(['references', 'showActions' => true, 'showCheckboxes' => false, 'selectedReferences' => [], 'mode' => 'simple'])

@php
    // Meses em português (por número)
    $monthsPortuguese = [
        1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
        5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
        9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
    ];

    // Mapeamento de meses em inglês para número
    $monthsEnglishToNumber = [
        'january' => 1, 'february' => 2, 'march' => 3, 'april' => 4,
        'may' => 5, 'june' => 6, 'july' => 7, 'august' => 8,
        'september' => 9, 'october' => 10, 'november' => 11, 'december' => 12
    ];

    // Mapeamento de meses em português para número
    $monthsPortugueseToNumber = [
        'janeiro' => 1, 'fevereiro' => 2, 'março' => 3, 'abril' => 4,
        'maio' => 5, 'junho' => 6, 'julho' => 7, 'agosto' => 8,
        'setembro' => 9, 'outubro' => 10, 'novembro' => 11, 'dezembro' => 12
    ];

    // Função para converter fee_month para número (suporta inglês E português)
    $getMonthNumber = function($feeMonth) use ($monthsEnglishToNumber, $monthsPortugueseToNumber) {
        if (is_numeric($feeMonth)) {
            return (int)$feeMonth;
        }
        $monthLower = strtolower(trim($feeMonth));

        // Tentar inglês primeiro
        if (isset($monthsEnglishToNumber[$monthLower])) {
            return $monthsEnglishToNumber[$monthLower];
        }

        // Tentar português
        if (isset($monthsPortugueseToNumber[$monthLower])) {
            return $monthsPortugueseToNumber[$monthLower];
        }

        // Fallback
        return 1;
    };

    // Agrupar referências por mês/ano e ordenar de Janeiro a Dezembro
    $groupedReferences = collect($references)->groupBy(function($reference) {
        return $reference->fee_month . '-' . $reference->fee_year;
    })->sortBy(function($group, $key) use ($getMonthNumber) {
        list($month, $year) = explode('-', $key);
        $monthNum = $getMonthNumber($month);
        // Formato: YYYYMM (ex: 202501 = Janeiro 2025, 202512 = Dezembro 2025)
        return $year . str_pad($monthNum, 2, '0', STR_PAD_LEFT);
    });
@endphp

<style>
    .month-group-card {
        border-radius: 12px;
        overflow: hidden;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        margin-bottom: 1.5rem;
        border: 1px solid #e0e0e0;
    }

    .month-group-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 1rem 1.5rem;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .month-group-header h5 {
        margin: 0;
        font-size: 1.25rem;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 0.75rem;
    }

    .month-badge {
        background: rgba(255, 255, 255, 0.25);
        padding: 0.4rem 0.9rem;
        border-radius: 20px;
        font-size: 0.9rem;
        font-weight: 600;
        backdrop-filter: blur(10px);
    }

    .reference-item {
        border-bottom: 1px solid #f0f0f0;
        padding: 1.25rem 1.5rem;
        transition: all 0.2s ease;
        background: white;
    }

    .reference-item:last-child {
        border-bottom: none;
    }

    .reference-item:hover {
        background: #f8f9fa;
        transform: translateX(5px);
    }

    .reference-item.paid {
        background: linear-gradient(to right, #e8f5e9 0%, #ffffff 100%);
        border-left: 4px solid #4caf50;
    }

    .reference-item.pending {
        background: linear-gradient(to right, #fff8e1 0%, #ffffff 100%);
        border-left: 4px solid #ff9800;
    }

    .reference-item.expired {
        background: linear-gradient(to right, #ffebee 0%, #ffffff 100%);
        border-left: 4px solid #f44336;
    }

    .reference-item.approved {
        background: linear-gradient(to right, #e3f2fd 0%, #ffffff 100%);
        border-left: 4px solid #2196f3;
    }

    .reference-number {
        font-family: 'Courier New', monospace;
        font-size: 1.1rem;
        font-weight: 700;
        color: #1976d2;
        letter-spacing: 1px;
    }

    .reference-amount {
        font-size: 1.5rem;
        font-weight: 700;
        color: #2c3e50;
    }

    .reference-description {
        color: #546e7a;
        font-size: 0.95rem;
        font-weight: 500;
    }

    .status-badge {
        padding: 0.4rem 1rem;
        border-radius: 20px;
        font-size: 0.85rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .status-badge.paid {
        background: linear-gradient(135deg, #66bb6a 0%, #4caf50 100%);
        color: white;
    }

    .status-badge.pending {
        background: linear-gradient(135deg, #ffa726 0%, #fb8c00 100%);
        color: white;
    }

    .status-badge.expired {
        background: linear-gradient(135deg, #ef5350 0%, #e53935 100%);
        color: white;
    }

    .status-badge.approved {
        background: linear-gradient(135deg, #42a5f5 0%, #1976d2 100%);
        color: white;
    }

    .action-btn {
        padding: 0.5rem 1.25rem;
        border-radius: 8px;
        font-weight: 600;
        font-size: 0.9rem;
        transition: all 0.2s ease;
        border: none;
    }

    .action-btn.pay {
        background: linear-gradient(135deg, #66bb6a 0%, #4caf50 100%);
        color: white;
    }

    .action-btn.pay:hover {
        background: linear-gradient(135deg, #4caf50 0%, #388e3c 100%);
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(76, 175, 80, 0.4);
    }

    .action-btn.view {
        background: linear-gradient(135deg, #42a5f5 0%, #1976d2 100%);
        color: white;
    }

    .action-btn.view:hover {
        background: linear-gradient(135deg, #1976d2 0%, #1565c0 100%);
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(25, 118, 210, 0.4);
    }

    .reference-meta {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        color: #78909c;
        font-size: 0.85rem;
    }

    .empty-state {
        text-align: center;
        padding: 4rem 2rem;
        background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
        border-radius: 12px;
    }

    .empty-state-icon {
        font-size: 5rem;
        color: #90a4ae;
        opacity: 0.5;
        margin-bottom: 1rem;
    }
</style>

@if($groupedReferences->isEmpty())
    <div class="empty-state">
        <div class="empty-state-icon">
            <i class="fas fa-receipt"></i>
        </div>
        <h4 class="text-muted">Nenhuma referência encontrada</h4>
        <p class="text-muted mb-0">Não há referências de pagamento disponíveis no momento.</p>
    </div>
@else
    @foreach($groupedReferences as $monthYearKey => $monthReferences)
        @php
            $firstRef = $monthReferences->first();
            $monthNumber = $getMonthNumber($firstRef->fee_month);
            $monthName = $monthsPortuguese[$monthNumber] ?? 'Mês ' . $monthNumber;
            $year = $firstRef->fee_year;

            // Calcular totais do mês
            $totalAmount = $monthReferences->sum('amount');
            $paidAmount = $monthReferences->where('status', 'paid')->sum('amount');
            $pendingAmount = $monthReferences->where('status', 'pending')->sum('amount');
        @endphp

        <div class="month-group-card">
            <div class="month-group-header">
                <h5>
                    <i class="fas fa-calendar-alt"></i>
                    {{ $monthName }} / {{ $year }}
                </h5>
                <div class="d-flex gap-2 align-items-center">
                    <span class="month-badge">
                        {{ $monthReferences->count() }} {{ $monthReferences->count() === 1 ? 'referência' : 'referências' }}
                    </span>
                    <span class="month-badge">
                        Total: {{ number_format($totalAmount, 2) }} MT
                    </span>
                </div>
            </div>

            @if($mode === 'detailed')
            <!-- Modo Tabela Detalhada (Admin) -->
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead style="background: #f8f9fa;">
                        <tr>
                            @if($showCheckboxes)
                            <th style="width: 40px;"></th>
                            @endif
                            <th>Referência</th>
                            <th>Descrição</th>
                            <th>Vencimento</th>
                            <th>Estado</th>
                            <th class="text-end">Valor Base</th>
                            <th class="text-end">Multa</th>
                            <th class="text-end">Desconto</th>
                            <th class="text-end">Total</th>
                            <th>Data/Hora</th>
                            @if($showActions)
                            <th class="text-center">Ações</th>
                            @endif
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($monthReferences as $reference)
                        @php
                            $baseAmount = $reference->amount - ($reference->fine_amount ?? 0);
                            $fineAmount = $reference->fine_amount ?? 0;
                            $discountAmount = $reference->discount_amount ?? 0;

                            // Determinar status para filtro
                            $dataStatus = 'pending';
                            if ($reference->status === 'paid') {
                                $dataStatus = 'paid';
                            } elseif ($reference->approval_status === 'approved') {
                                $dataStatus = 'approved';
                            } elseif (isset($reference->expires_at) && $reference->expires_at < now()) {
                                $dataStatus = 'expired';
                            }
                        @endphp
                        <tr data-status="{{ $dataStatus }}">
                            @if($showCheckboxes)
                            <td>
                                <input class="form-check-input" type="checkbox" wire:model="selectedReferences" value="{{ $reference->id }}">
                            </td>
                            @endif
                            <td>
                                <span class="badge bg-primary font-monospace">{{ $reference->reference_number }}</span>
                            </td>
                            <td>
                                @if(isset($reference->metadata['description']))
                                    {{ $reference->metadata['description'] }}
                                @elseif(isset($reference->metadata['custom_fee_type']))
                                    {{ $reference->metadata['custom_fee_type'] }}
                                @else
                                    Propina {{ $monthName }}/{{ $year }}
                                @endif
                            </td>
                            <td>
                                @if(isset($reference->expires_at) && $reference->expires_at)
                                    @php
                                        $expiryDate = $reference->expires_at;
                                        if (is_string($expiryDate)) {
                                            $expiryDate = \Carbon\Carbon::parse($expiryDate);
                                        }
                                    @endphp
                                    {{ $expiryDate->format('d/m/Y') }}
                                @else
                                    N/A
                                @endif
                            </td>
                            <td>
                                @if($reference->status === 'paid')
                                    <span class="badge bg-success"><i class="fas fa-check me-1"></i>Pago</span>
                                @elseif($reference->status === 'overdue')
                                    <span class="badge bg-danger"><i class="fas fa-exclamation-triangle me-1"></i>Em Atraso</span>
                                @elseif($reference->approval_status === 'approved')
                                    <span class="badge bg-info"><i class="fas fa-check me-1"></i>Aprovado</span>
                                @elseif(isset($reference->expires_at) && $reference->expires_at < now())
                                    <span class="badge bg-danger"><i class="fas fa-times me-1"></i>Expirado</span>
                                @else
                                    <span class="badge bg-warning text-dark"><i class="fas fa-clock me-1"></i>Pendente</span>
                                @endif
                            </td>
                            <td class="text-end">{{ number_format($baseAmount, 2) }} MT</td>
                            <td class="text-end">{{ number_format($fineAmount, 2) }} MT</td>
                            <td class="text-end">{{ number_format($discountAmount, 2) }} MT</td>
                            <td class="text-end fw-bold">{{ number_format($reference->amount, 2) }} MT</td>
                            <td>
                                @if($reference->status === 'paid' && isset($reference->paid_at))
                                    {{ $reference->paid_at->format('d/m/Y H:i') }}
                                @else
                                    {{ $reference->created_at->format('d/m/Y H:i') }}
                                @endif
                            </td>
                            @if($showActions)
                            <td class="text-center">
                                @php
                                    // Verificar se a referência expirou
                                    $isExpired = false;
                                    if ($reference->status === 'pending' && isset($reference->expires_at)) {
                                        try {
                                            $expiryDate = is_string($reference->expires_at)
                                                ? \Carbon\Carbon::parse($reference->expires_at)
                                                : $reference->expires_at;
                                            $isExpired = $expiryDate->isPast();
                                        } catch (\Exception $e) {
                                            $isExpired = false;
                                        }
                                    }
                                @endphp

                                <button class="btn btn-sm btn-outline-primary" wire:click="viewReferenceById('{{ $reference->id }}')" title="Ver detalhes">
                                    <i class="fas fa-eye"></i>
                                </button>

                                @if($reference->status === 'paid')
                                    {{-- Botões para pagamentos pagos --}}
                                    @if(str_starts_with($reference->id, 'fee_'))
                                        {{-- Pagamento direto (fee_assign) - apenas recibo --}}
                                        @php
                                            $feeAssignId = str_replace('fee_', '', $reference->id);
                                        @endphp
                                        <button class="btn btn-sm btn-success ms-1" wire:click="downloadPaymentReceipt({{ $feeAssignId }})" title="Baixar Recibo">
                                            <i class="fas fa-file-invoice"></i>
                                        </button>
                                    @else
                                        {{-- PaymentReference normal - comprovativo + recibo --}}
                                        <button class="btn btn-sm btn-info ms-1" wire:click="downloadReceipt('{{ $reference->id }}')" title="Baixar Comprovativo">
                                            <i class="fas fa-file-download"></i>
                                        </button>
                                        <button class="btn btn-sm btn-success ms-1" wire:click="downloadOfficialReceipt('{{ $reference->id }}')" title="Baixar Recibo">
                                            <i class="fas fa-file-invoice"></i>
                                        </button>
                                    @endif
                                @elseif($isExpired)
                                    {{-- Referência expirada - verificar se estudante pode regenerar --}}
                                    @php
                                        $regenerationCount = $reference->metadata['regeneration_count'] ?? 0;
                                        $feeStructureId = $reference->metadata['fee_structure_id'] ?? null;
                                        $pendingBase = $reference->metadata['base_amount'] ?? ($reference->amount - ($reference->fine_amount ?? 0));

                                        // Estudante pode regenerar se:
                                        // 1. Nunca regenerou (regeneration_count = 0)
                                        // 2. Tem fee_structure_id (é uma mensalidade)
                                        $studentCanRegenerate = $regenerationCount < 1 && $feeStructureId;
                                    @endphp

                                    @if($mode === 'detailed' && auth('admin')->check())
                                        {{-- Admin pode sempre regenerar --}}
                                        <button class="btn btn-sm btn-warning ms-1"
                                                onclick="openPaymentModal('{{ $monthName }}', {{ $year }}, {{ $pendingBase }}, 0, 0, {{ $feeStructureId ?? 'null' }})"
                                                title="Gerar Nova Referência (Admin)">
                                            <i class="fas fa-redo me-1"></i>Regenerar
                                        </button>
                                    @elseif($studentCanRegenerate)
                                        {{-- Estudante pode regenerar (1ª vez expirou, ainda não regenerou) --}}
                                        <button class="btn btn-sm btn-warning ms-1"
                                                onclick="openPaymentModal('{{ $monthName }}', {{ $year }}, {{ $pendingBase }}, 0, 0, {{ $feeStructureId ?? 'null' }})"
                                                title="Regenerar referência expirada (inclui multa actualizada)">
                                            <i class="fas fa-redo me-1"></i>Regenerar
                                        </button>
                                        <small class="d-block text-muted mt-1" style="font-size: 0.7rem;">
                                            <i class="fas fa-info-circle"></i> Última oportunidade
                                        </small>
                                    @else
                                        {{-- Estudante já usou sua regeneração ou não tem fee_structure_id --}}
                                        <span class="badge bg-secondary" title="Já utilizou sua opção de regenerar. Contacte o administrador.">
                                            <i class="fas fa-lock me-1"></i>Contacte Admin
                                        </span>
                                    @endif
                                @elseif($reference->status === 'overdue' || (isset($reference->metadata['is_pending_structure']) && $reference->metadata['is_pending_structure']))
                                    {{-- Mensalidade em atraso ou pendente das FeeStructures - botão para pagar --}}
                                    @php
                                        $pendingBase = $reference->metadata['base_amount'] ?? ($reference->amount - ($reference->fine_amount ?? 0));
                                        $pendingFine = $reference->metadata['fine_amount'] ?? $reference->fine_amount ?? 0;
                                        $feeStructureId = $reference->metadata['fee_structure_id'] ?? null;
                                    @endphp
                                    <button class="btn btn-sm btn-primary ms-1"
                                            onclick="openPaymentModal('{{ $monthName }}', {{ $year }}, {{ $pendingBase }}, {{ $pendingFine }}, 0, {{ $feeStructureId ?? 'null' }})"
                                            title="Gerar Referência de Pagamento">
                                        <i class="fas fa-credit-card me-1"></i>Pagar
                                    </button>
                                @elseif($reference->approval_status === 'approved' || $reference->status === 'pending')
                                    {{-- Referência pendente válida - mostrar comprovativo --}}
                                    @if(!str_starts_with($reference->id, 'fee_') && !str_starts_with($reference->id, 'pending_'))
                                        <button class="btn btn-sm btn-info ms-1" wire:click="downloadReceipt('{{ $reference->id }}')" title="Baixar Comprovativo">
                                            <i class="fas fa-file-download"></i>
                                        </button>
                                    @endif
                                @endif
                            </td>
                            @endif
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
            @else
            <!-- Modo Simples (Cards) -->
            <div>
                @foreach($monthReferences as $reference)
                    @php
                        $statusClass = $reference->status ?? 'pending';
                        if ($reference->approval_status === 'approved') {
                            $statusClass = 'approved';
                        } elseif (isset($reference->expires_at) && $reference->expires_at < now() && $reference->status !== 'paid') {
                            $statusClass = 'expired';
                        }
                    @endphp

                    <div class="reference-item {{ $statusClass }}">
                        <div class="row align-items-center">
                            @if($showCheckboxes)
                            <div class="col-auto">
                                <input class="form-check-input" type="checkbox" wire:model="selectedReferences" value="{{ $reference->id }}" style="width: 1.25rem; height: 1.25rem;">
                            </div>
                            @endif

                            <!-- Informação Principal -->
                            <div class="col-md-4">
                                <div class="reference-description mb-2">
                                    @if(isset($reference->metadata['description']))
                                        {{ $reference->metadata['description'] }}
                                    @elseif(isset($reference->metadata['custom_fee_type']))
                                        {{ $reference->metadata['custom_fee_type'] }}
                                    @elseif(isset($reference->metadata['pos_product_name']))
                                        {{ $reference->metadata['pos_product_name'] }}
                                    @else
                                        Propina {{ $monthName }}/{{ $year }}
                                    @endif
                                </div>
                                <div class="reference-meta">
                                    <i class="fas fa-hashtag"></i>
                                    <span class="reference-number">{{ $reference->reference_number }}</span>
                                </div>
                            </div>

                            <!-- Valor -->
                            <div class="col-md-2 text-center">
                                <div class="reference-amount">
                                    {{ number_format($reference->amount, 2) }}
                                    <small class="d-block text-muted" style="font-size: 0.7rem;">MT</small>
                                </div>
                            </div>

                            <!-- Validade -->
                            <div class="col-md-2">
                                @if(isset($reference->expires_at))
                                <div class="text-muted">
                                    <i class="fas fa-clock me-1"></i>
                                    <small>Expira em</small>
                                    <div class="fw-bold">{{ $reference->expires_at->format('d/m/Y') }}</div>
                                </div>
                                @endif
                            </div>

                            <!-- Status -->
                            <div class="col-md-2 text-center">
                                @if($reference->status === 'paid')
                                    <span class="status-badge paid">
                                        <i class="fas fa-check-circle me-1"></i>Pago
                                    </span>
                                @elseif($reference->approval_status === 'approved')
                                    <span class="status-badge approved">
                                        <i class="fas fa-check me-1"></i>Aprovado
                                    </span>
                                @elseif(isset($reference->expires_at) && $reference->expires_at < now())
                                    <span class="status-badge expired">
                                        <i class="fas fa-times-circle me-1"></i>Expirado
                                    </span>
                                @else
                                    <span class="status-badge pending">
                                        <i class="fas fa-hourglass-half me-1"></i>Pendente
                                    </span>
                                @endif
                            </div>

                            <!-- Ações -->
                            @if($showActions)
                            <div class="col-md-2 text-end">
                                @php
                                    // Verificar se expirou (modo cards)
                                    $isExpiredCards = false;
                                    if ($reference->status !== 'paid' && isset($reference->expires_at)) {
                                        try {
                                            $expiryDateCards = is_string($reference->expires_at)
                                                ? \Carbon\Carbon::parse($reference->expires_at)
                                                : $reference->expires_at;
                                            $isExpiredCards = $expiryDateCards->isPast();
                                        } catch (\Exception $e) {
                                            $isExpiredCards = false;
                                        }
                                    }

                                    $regenerationCountCards = $reference->metadata['regeneration_count'] ?? 0;
                                    $feeStructureIdCards = $reference->metadata['fee_structure_id'] ?? null;
                                    $pendingBaseCards = $reference->metadata['base_amount'] ?? ($reference->amount - ($reference->fine_amount ?? 0));
                                    $studentCanRegenerateCards = $regenerationCountCards < 1 && $feeStructureIdCards;
                                @endphp

                                @if($reference->status === 'paid')
                                    {{-- Botão de recibo para pagamentos pagos --}}
                                    @if(str_starts_with($reference->id, 'fee_'))
                                        {{-- Pagamento direto (fee_assign) --}}
                                        @php
                                            $feeAssignId = str_replace('fee_', '', $reference->id);
                                        @endphp
                                        <button class="action-btn pay" wire:click="downloadPaymentReceipt({{ $feeAssignId }})" title="Baixar Recibo">
                                            <i class="fas fa-file-invoice me-1"></i>Recibo
                                        </button>
                                    @else
                                        {{-- PaymentReference normal --}}
                                        <button class="action-btn pay" wire:click="downloadOfficialReceipt('{{ $reference->id }}')" title="Baixar Recibo">
                                            <i class="fas fa-file-invoice me-1"></i>Recibo
                                        </button>
                                    @endif
                                @elseif($isExpiredCards && $studentCanRegenerateCards)
                                    {{-- Referência expirada - estudante pode regenerar --}}
                                    <button class="action-btn" style="background: linear-gradient(135deg, #ffa726 0%, #fb8c00 100%); color: white;"
                                            onclick="openPaymentModal('{{ $monthName }}', {{ $year }}, {{ $pendingBaseCards }}, 0, 0, {{ $feeStructureIdCards ?? 'null' }})"
                                            title="Regenerar (última oportunidade)">
                                        <i class="fas fa-redo me-1"></i>Regenerar
                                    </button>
                                @elseif($isExpiredCards)
                                    {{-- Referência expirada - estudante já usou regeneração --}}
                                    <span class="badge bg-secondary" title="Contacte o administrador">
                                        <i class="fas fa-lock me-1"></i>Contacte Admin
                                    </span>
                                @elseif($reference->approval_status === 'approved')
                                    <button class="action-btn pay" onclick="payReference('{{ $reference->id }}')">
                                        <i class="fas fa-credit-card me-1"></i>Pagar
                                    </button>
                                @else
                                    <button class="action-btn view" onclick="viewReference('{{ $reference->id }}')">
                                        <i class="fas fa-eye me-1"></i>Ver
                                    </button>
                                @endif
                            </div>
                            @endif
                        </div>
                    </div>
                @endforeach
            </div>
            @endif
        </div>
    @endforeach
@endif
